<?php
require_once '../config/database.php';
require_once '../config/auth.php';
require_once '../config/email.php';
requireLogin('admin', 'index.php');

$db = Database::getInstance()->getConnection();
$sucesso = '';
$erro = '';

// Ações
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $entregadorId = intval($_POST['entregador_id'] ?? 0);
    
    // Aprovar entregador
    if (isset($_POST['aprovar'])) {
        $stmt = $db->prepare("UPDATE entregadores SET aprovado = 'aprovado', data_aprovacao = NOW() WHERE id = ?");
        $stmt->execute([$entregadorId]);
        
        // Buscar dados para enviar email
        $stmt = $db->prepare("SELECT email, nome FROM entregadores WHERE id = ?");
        $stmt->execute([$entregadorId]);
        $ent = $stmt->fetch();
        if ($ent) {
            enviarEmailAprovacao($ent['email'], $ent['nome']);
        }
        $sucesso = 'Entregador aprovado com sucesso! Email de confirmação enviado.';
    }
    
    // Rejeitar entregador
    if (isset($_POST['rejeitar'])) {
        $motivo = trim($_POST['motivo_rejeicao'] ?? 'Documentação incompleta ou inválida');
        $stmt = $db->prepare("UPDATE entregadores SET aprovado = 'rejeitado', motivo_rejeicao = ? WHERE id = ?");
        $stmt->execute([$motivo, $entregadorId]);
        
        // Buscar dados para enviar email
        $stmt = $db->prepare("SELECT email, nome FROM entregadores WHERE id = ?");
        $stmt->execute([$entregadorId]);
        $ent = $stmt->fetch();
        if ($ent) {
            enviarEmailRejeicao($ent['email'], $ent['nome'], $motivo);
        }
        $sucesso = 'Entregador rejeitado. Email enviado com o motivo.';
    }
    
    // Bloquear
    if (isset($_POST['bloquear'])) {
        $motivo = trim($_POST['motivo'] ?? '');
        $stmt = $db->prepare("UPDATE entregadores SET bloqueado = 1, motivo_bloqueio = ?, status = 'offline' WHERE id = ?");
        $stmt->execute([$motivo, $entregadorId]);
        $sucesso = 'Entregador bloqueado com sucesso!';
    }
    
    // Desbloquear
    if (isset($_POST['desbloquear'])) {
        $stmt = $db->prepare("UPDATE entregadores SET bloqueado = 0, motivo_bloqueio = NULL WHERE id = ?");
        $stmt->execute([$entregadorId]);
        $sucesso = 'Entregador desbloqueado com sucesso!';
    }
}

// Filtro
$filtro = $_GET['filtro'] ?? 'todos';
$where = '';
if ($filtro === 'pendentes') $where = "WHERE e.aprovado = 'pendente'";
elseif ($filtro === 'aprovados') $where = "WHERE e.aprovado = 'aprovado'";
elseif ($filtro === 'rejeitados') $where = "WHERE e.aprovado = 'rejeitado'";
elseif ($filtro === 'bloqueados') $where = "WHERE e.bloqueado = 1";

// Buscar entregadores
$stmt = $db->query("
    SELECT e.*, 
           (SELECT COUNT(*) FROM entregas WHERE entregador_id = e.id AND status = 'entregue') as total_entregas,
           (SELECT COALESCE(SUM(valor), 0) FROM entregas WHERE entregador_id = e.id AND status = 'entregue') as total_ganhos
    FROM entregadores e
    $where
    ORDER BY 
        CASE WHEN e.aprovado = 'pendente' THEN 0 ELSE 1 END,
        e.created_at DESC
");
$entregadores = $stmt->fetchAll(PDO::FETCH_ASSOC);

// Contadores
$stmt = $db->query("SELECT COUNT(*) FROM entregadores WHERE aprovado = 'pendente'");
$totalPendentes = $stmt->fetchColumn();

$pageTitle = 'Entregadores';
include 'includes/header.php';
?>
    
    <div class="container">
        <h2 style="margin-bottom: 20px;">🏍️ Entregadores</h2>
        
        <?php if ($sucesso): ?>
            <div class="alert alert-success"><?= htmlspecialchars($sucesso) ?></div>
        <?php endif; ?>
        
        <!-- Filtros -->
        <div class="card" style="margin-bottom: 20px; padding: 15px;">
            <div style="display: flex; gap: 10px; flex-wrap: wrap; align-items: center;">
                <span style="font-weight: 500;">Filtrar:</span>
                <a href="?filtro=todos" class="btn btn-sm <?= $filtro === 'todos' ? 'btn-primary' : '' ?>">Todos</a>
                <a href="?filtro=pendentes" class="btn btn-sm <?= $filtro === 'pendentes' ? 'btn-primary' : '' ?>" style="position: relative;">
                    ⏳ Pendentes
                    <?php if ($totalPendentes > 0): ?>
                        <span style="position: absolute; top: -8px; right: -8px; background: #ff1744; color: white; font-size: 11px; padding: 2px 6px; border-radius: 10px;"><?= $totalPendentes ?></span>
                    <?php endif; ?>
                </a>
                <a href="?filtro=aprovados" class="btn btn-sm <?= $filtro === 'aprovados' ? 'btn-primary' : '' ?>">✅ Aprovados</a>
                <a href="?filtro=rejeitados" class="btn btn-sm <?= $filtro === 'rejeitados' ? 'btn-primary' : '' ?>">❌ Rejeitados</a>
                <a href="?filtro=bloqueados" class="btn btn-sm <?= $filtro === 'bloqueados' ? 'btn-primary' : '' ?>">🚫 Bloqueados</a>
            </div>
        </div>
        
        <?php if (empty($entregadores)): ?>
            <div class="card" style="text-align: center; padding: 60px;">
                <div style="font-size: 60px; margin-bottom: 20px;">📭</div>
                <h3>Nenhum entregador encontrado</h3>
                <p style="color: var(--text-muted);">Não há entregadores com este filtro.</p>
            </div>
        <?php else: ?>
            <?php foreach ($entregadores as $e): ?>
                <div class="card" style="margin-bottom: 15px; <?= $e['aprovado'] === 'pendente' ? 'border-left: 4px solid #ff9800;' : '' ?>">
                    <div style="display: flex; gap: 20px; flex-wrap: wrap;">
                        <!-- Info Principal -->
                        <div style="flex: 1; min-width: 250px;">
                            <div style="display: flex; align-items: center; gap: 10px; margin-bottom: 10px;">
                                <span style="font-size: 30px;">
                                    <?php
                                    $veiculos = ['moto' => '🏍️', 'carro' => '🚗', 'bicicleta' => '🚲', 'van' => '🚐', 'caminhao' => '🚛'];
                                    echo $veiculos[$e['veiculo'] ?? 'moto'] ?? '🏍️';
                                    ?>
                                </span>
                                <div>
                                    <strong style="font-size: 16px;"><?= htmlspecialchars($e['nome']) ?></strong>
                                    <div style="font-size: 13px; color: var(--text-muted);">#<?= $e['id'] ?></div>
                                </div>
                                
                                <!-- Status Badge -->
                                <?php if ($e['aprovado'] === 'pendente'): ?>
                                    <span style="background: #fff3e0; color: #e65100; padding: 4px 10px; border-radius: 20px; font-size: 12px; font-weight: 500;">⏳ Pendente</span>
                                <?php elseif ($e['aprovado'] === 'rejeitado'): ?>
                                    <span style="background: #ffebee; color: #c62828; padding: 4px 10px; border-radius: 20px; font-size: 12px; font-weight: 500;">❌ Rejeitado</span>
                                <?php elseif (!empty($e['bloqueado'])): ?>
                                    <span style="background: #ffebee; color: #c62828; padding: 4px 10px; border-radius: 20px; font-size: 12px; font-weight: 500;">🚫 Bloqueado</span>
                                <?php else: ?>
                                    <span style="background: #e8f5e9; color: #2e7d32; padding: 4px 10px; border-radius: 20px; font-size: 12px; font-weight: 500;">✅ Aprovado</span>
                                <?php endif; ?>
                            </div>
                            
                            <div style="display: grid; grid-template-columns: repeat(auto-fit, minmax(150px, 1fr)); gap: 8px; font-size: 14px;">
                                <div>📧 <?= htmlspecialchars($e['email']) ?></div>
                                <div>📱 <?= htmlspecialchars($e['telefone'] ?? '-') ?></div>
                                <div>📋 CPF: <?= htmlspecialchars($e['cpf'] ?? '-') ?></div>
                                <?php if (!empty($e['placa'])): ?>
                                    <div>🚗 Placa: <?= htmlspecialchars($e['placa']) ?></div>
                                <?php endif; ?>
                                <?php if (!empty($e['veiculo_marca'])): ?>
                                    <div>🏭 <?= htmlspecialchars($e['veiculo_marca'] . ' ' . ($e['veiculo_modelo'] ?? '')) ?></div>
                                <?php endif; ?>
                            </div>
                            
                            <?php if (!empty($e['endereco'])): ?>
                                <div style="margin-top: 8px; font-size: 13px; color: var(--text-muted);">
                                    📍 <?= htmlspecialchars($e['endereco']) ?>
                                </div>
                            <?php endif; ?>
                            
                            <div style="margin-top: 10px; font-size: 13px; color: var(--text-muted);">
                                📅 Cadastro: <?= date('d/m/Y H:i', strtotime($e['created_at'])) ?>
                                <?php if ($e['aprovado'] === 'aprovado' && !empty($e['data_aprovacao'])): ?>
                                    | ✅ Aprovado em: <?= date('d/m/Y', strtotime($e['data_aprovacao'])) ?>
                                <?php endif; ?>
                            </div>
                        </div>
                        
                        <!-- Estatísticas -->
                        <div style="text-align: center; padding: 10px 20px; background: var(--bg-primary); border-radius: 8px;">
                            <div style="font-size: 24px; font-weight: bold; color: var(--primary);"><?= $e['total_entregas'] ?></div>
                            <div style="font-size: 12px; color: var(--text-muted);">Entregas</div>
                            <div style="font-size: 16px; font-weight: 500; color: #00c853; margin-top: 5px;">
                                R$ <?= number_format($e['total_ganhos'], 2, ',', '.') ?>
                            </div>
                        </div>
                    </div>
                    
                    <!-- Documentos -->
                    <div style="margin-top: 15px; padding-top: 15px; border-top: 1px solid var(--border-color);">
                        <strong style="font-size: 13px;">📄 Documentos:</strong>
                        <div style="display: flex; gap: 10px; flex-wrap: wrap; margin-top: 8px;">
                            <?php
                            $docs = [
                                'doc_identidade' => '🪪 RG',
                                'doc_cpf' => '📄 CPF',
                                'doc_cnh' => '🚗 CNH',
                                'doc_veiculo' => '📋 CRLV',
                                'doc_endereco' => '🏠 Endereço'
                            ];
                            foreach ($docs as $campo => $label):
                                if (!empty($e[$campo])):
                            ?>
                                <a href="../<?= htmlspecialchars($e[$campo]) ?>" target="_blank" 
                                   style="display: inline-flex; align-items: center; gap: 5px; padding: 6px 12px; background: var(--bg-primary); border-radius: 6px; text-decoration: none; color: var(--text-primary); font-size: 13px; border: 1px solid var(--border-color);">
                                    <?= $label ?> 👁️
                                </a>
                            <?php 
                                endif;
                            endforeach; 
                            ?>
                            
                            <?php if (empty($e['doc_identidade']) && empty($e['doc_cpf'])): ?>
                                <span style="color: var(--text-muted); font-size: 13px;">Nenhum documento enviado</span>
                            <?php endif; ?>
                        </div>
                    </div>
                    
                    <!-- Ações -->
                    <div style="margin-top: 15px; padding-top: 15px; border-top: 1px solid var(--border-color); display: flex; gap: 10px; flex-wrap: wrap;">
                        <?php if ($e['aprovado'] === 'pendente'): ?>
                            <form method="POST" style="display: inline;">
                                <input type="hidden" name="entregador_id" value="<?= $e['id'] ?>">
                                <button type="submit" name="aprovar" class="btn btn-success btn-sm" onclick="return confirm('Aprovar este entregador?')">
                                    ✅ Aprovar
                                </button>
                            </form>
                            <button type="button" class="btn btn-danger btn-sm" onclick="abrirModalRejeitar(<?= $e['id'] ?>, '<?= htmlspecialchars($e['nome'], ENT_QUOTES) ?>')">
                                ❌ Rejeitar
                            </button>
                        <?php elseif ($e['aprovado'] === 'aprovado'): ?>
                            <?php if (!empty($e['bloqueado'])): ?>
                                <form method="POST" style="display: inline;">
                                    <input type="hidden" name="entregador_id" value="<?= $e['id'] ?>">
                                    <button type="submit" name="desbloquear" class="btn btn-success btn-sm">
                                        🔓 Desbloquear
                                    </button>
                                </form>
                            <?php else: ?>
                                <button type="button" class="btn btn-danger btn-sm" onclick="abrirModalBloquear(<?= $e['id'] ?>, '<?= htmlspecialchars($e['nome'], ENT_QUOTES) ?>')">
                                    🚫 Bloquear
                                </button>
                            <?php endif; ?>
                        <?php elseif ($e['aprovado'] === 'rejeitado'): ?>
                            <form method="POST" style="display: inline;">
                                <input type="hidden" name="entregador_id" value="<?= $e['id'] ?>">
                                <button type="submit" name="aprovar" class="btn btn-success btn-sm" onclick="return confirm('Aprovar este entregador?')">
                                    ✅ Aprovar Agora
                                </button>
                            </form>
                            <?php if (!empty($e['motivo_rejeicao'])): ?>
                                <span style="font-size: 13px; color: var(--text-muted);">
                                    Motivo: <?= htmlspecialchars($e['motivo_rejeicao']) ?>
                                </span>
                            <?php endif; ?>
                        <?php endif; ?>
                    </div>
                </div>
            <?php endforeach; ?>
        <?php endif; ?>
    </div>
    
    <!-- Modal Rejeitar -->
    <div class="modal" id="modal-rejeitar">
        <div class="modal-content">
            <h3>❌ Rejeitar Entregador</h3>
            <p style="margin-bottom: 16px; color: var(--text-secondary);">
                Rejeitar cadastro de <strong id="nome-rejeitar"></strong>?
            </p>
            <form method="POST">
                <input type="hidden" name="entregador_id" id="rejeitar-entregador-id">
                <div class="form-group">
                    <label>Motivo da rejeição *</label>
                    <textarea name="motivo_rejeicao" class="form-control" rows="3" required placeholder="Ex: Documentos ilegíveis, CNH vencida..."></textarea>
                </div>
                <div style="display: flex; gap: 10px;">
                    <button type="submit" name="rejeitar" class="btn btn-danger">❌ Confirmar Rejeição</button>
                    <button type="button" class="btn" style="background: var(--border-color);" onclick="fecharModal('modal-rejeitar')">Cancelar</button>
                </div>
            </form>
        </div>
    </div>
    
    <!-- Modal Bloquear -->
    <div class="modal" id="modal-bloquear">
        <div class="modal-content">
            <h3>🚫 Bloquear Entregador</h3>
            <p style="margin-bottom: 16px; color: var(--text-secondary);">
                Bloquear <strong id="nome-bloquear"></strong>?
            </p>
            <form method="POST">
                <input type="hidden" name="entregador_id" id="bloquear-entregador-id">
                <div class="form-group">
                    <label>Motivo do bloqueio (opcional)</label>
                    <textarea name="motivo" class="form-control" rows="3" placeholder="Informe o motivo..."></textarea>
                </div>
                <div style="display: flex; gap: 10px;">
                    <button type="submit" name="bloquear" class="btn btn-danger">🚫 Confirmar Bloqueio</button>
                    <button type="button" class="btn" style="background: var(--border-color);" onclick="fecharModal('modal-bloquear')">Cancelar</button>
                </div>
            </form>
        </div>
    </div>
    
    <style>
        .modal {
            display: none;
            position: fixed;
            top: 0;
            left: 0;
            right: 0;
            bottom: 0;
            background: rgba(0,0,0,0.5);
            align-items: center;
            justify-content: center;
            z-index: 1000;
        }
        .modal.show { display: flex; }
        .modal-content {
            background: var(--card-bg);
            padding: 24px;
            border-radius: 12px;
            width: 100%;
            max-width: 450px;
            color: var(--text-primary);
        }
        .btn-sm {
            padding: 8px 14px;
            font-size: 13px;
        }
        .btn-success {
            background: #00c853;
            color: white;
        }
        .btn-danger {
            background: #ff1744;
            color: white;
        }
    </style>
    
    <script>
    function abrirModalRejeitar(id, nome) {
        document.getElementById('rejeitar-entregador-id').value = id;
        document.getElementById('nome-rejeitar').textContent = nome;
        document.getElementById('modal-rejeitar').classList.add('show');
    }
    
    function abrirModalBloquear(id, nome) {
        document.getElementById('bloquear-entregador-id').value = id;
        document.getElementById('nome-bloquear').textContent = nome;
        document.getElementById('modal-bloquear').classList.add('show');
    }
    
    function fecharModal(id) {
        document.getElementById(id).classList.remove('show');
    }
    
    // Fechar modal ao clicar fora
    document.querySelectorAll('.modal').forEach(modal => {
        modal.addEventListener('click', function(e) {
            if (e.target === this) {
                this.classList.remove('show');
            }
        });
    });
    </script>
</body>
</html>
