<?php
/**
 * Script de Atualização do Banco de Dados
 * Execute este arquivo para atualizar o banco sem perder dados
 */

require_once 'config/database.php';

$mensagens = [];
$erros = [];

try {
    $db = Database::getInstance()->getConnection();
    
    // 1. Criar tabela de empresas parceiras
    $db->exec("
        CREATE TABLE IF NOT EXISTS empresas_parceiras (
            id INT AUTO_INCREMENT PRIMARY KEY,
            nome_empresa VARCHAR(100) NOT NULL,
            cnpj VARCHAR(20) UNIQUE,
            email VARCHAR(100) NOT NULL,
            telefone VARCHAR(20),
            endereco VARCHAR(255),
            responsavel VARCHAR(100),
            status ENUM('pendente', 'aprovada', 'rejeitada') DEFAULT 'pendente',
            observacoes TEXT,
            created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
        )
    ");
    $mensagens[] = "✅ Tabela 'empresas_parceiras' verificada/criada";
    
    // 2. Criar tabela de configurações (se não existir)
    $db->exec("
        CREATE TABLE IF NOT EXISTS configuracoes (
            id INT AUTO_INCREMENT PRIMARY KEY,
            chave VARCHAR(50) UNIQUE NOT NULL,
            valor TEXT,
            descricao VARCHAR(255),
            updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP
        )
    ");
    $mensagens[] = "✅ Tabela 'configuracoes' verificada/criada";
    
    // 3. Criar tabela de notificações do cliente (se não existir)
    $db->exec("
        CREATE TABLE IF NOT EXISTS notificacoes_cliente (
            id INT AUTO_INCREMENT PRIMARY KEY,
            usuario_id INT NOT NULL,
            entrega_id INT NOT NULL,
            tipo ENUM('aceita', 'coletada', 'entregue', 'cancelada') NOT NULL,
            mensagem VARCHAR(255),
            lida TINYINT(1) DEFAULT 0,
            created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
        )
    ");
    $mensagens[] = "✅ Tabela 'notificacoes_cliente' verificada/criada";
    
    // 3.2 Atualizar ENUM da tabela notificacoes_cliente para incluir 'cancelada'
    try {
        $db->exec("ALTER TABLE notificacoes_cliente MODIFY COLUMN tipo ENUM('aceita', 'coletada', 'entregue', 'cancelada') NOT NULL");
        $mensagens[] = "✅ ENUM 'tipo' atualizado em 'notificacoes_cliente'";
    } catch (Exception $e) {
        // Já atualizado ou erro
    }
    
    // 3.1 Criar tabela de notificações para entregadores (se não existir)
    $db->exec("
        CREATE TABLE IF NOT EXISTS notificacoes (
            id INT AUTO_INCREMENT PRIMARY KEY,
            entregador_id INT,
            entrega_id INT,
            mensagem VARCHAR(255),
            lida TINYINT(1) DEFAULT 0,
            created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
        )
    ");
    $mensagens[] = "✅ Tabela 'notificacoes' verificada/criada";
    
    // 4. Adicionar colunas na tabela entregadores (se não existirem)
    $colunasEntregador = [
        'latitude' => "ALTER TABLE entregadores ADD COLUMN latitude DECIMAL(10, 8) NULL",
        'longitude' => "ALTER TABLE entregadores ADD COLUMN longitude DECIMAL(11, 8) NULL",
        'bloqueado' => "ALTER TABLE entregadores ADD COLUMN bloqueado TINYINT(1) DEFAULT 0",
        'motivo_bloqueio' => "ALTER TABLE entregadores ADD COLUMN motivo_bloqueio TEXT NULL",
        'aprovado' => "ALTER TABLE entregadores ADD COLUMN aprovado ENUM('pendente', 'aprovado', 'rejeitado') DEFAULT 'aprovado'",
        'motivo_rejeicao' => "ALTER TABLE entregadores ADD COLUMN motivo_rejeicao TEXT NULL",
        'data_aprovacao' => "ALTER TABLE entregadores ADD COLUMN data_aprovacao DATETIME NULL",
        'veiculo' => "ALTER TABLE entregadores ADD COLUMN veiculo ENUM('bicicleta', 'moto', 'carro', 'van', 'caminhao') DEFAULT 'moto'",
        'placa' => "ALTER TABLE entregadores ADD COLUMN placa VARCHAR(10) NULL",
        'cpf' => "ALTER TABLE entregadores ADD COLUMN cpf VARCHAR(14) NULL",
        'endereco' => "ALTER TABLE entregadores ADD COLUMN endereco VARCHAR(255) NULL",
        'veiculo_marca' => "ALTER TABLE entregadores ADD COLUMN veiculo_marca VARCHAR(50) NULL",
        'veiculo_modelo' => "ALTER TABLE entregadores ADD COLUMN veiculo_modelo VARCHAR(50) NULL",
        'veiculo_ano' => "ALTER TABLE entregadores ADD COLUMN veiculo_ano VARCHAR(4) NULL",
        'veiculo_cor' => "ALTER TABLE entregadores ADD COLUMN veiculo_cor VARCHAR(30) NULL",
        'doc_identidade' => "ALTER TABLE entregadores ADD COLUMN doc_identidade VARCHAR(255) NULL",
        'doc_cpf' => "ALTER TABLE entregadores ADD COLUMN doc_cpf VARCHAR(255) NULL",
        'doc_cnh' => "ALTER TABLE entregadores ADD COLUMN doc_cnh VARCHAR(255) NULL",
        'doc_veiculo' => "ALTER TABLE entregadores ADD COLUMN doc_veiculo VARCHAR(255) NULL",
        'doc_endereco' => "ALTER TABLE entregadores ADD COLUMN doc_endereco VARCHAR(255) NULL"
    ];
    
    foreach ($colunasEntregador as $coluna => $sql) {
        try {
            $stmt = $db->query("SHOW COLUMNS FROM entregadores LIKE '$coluna'");
            if ($stmt->rowCount() == 0) {
                $db->exec($sql);
                $mensagens[] = "✅ Coluna '$coluna' adicionada em 'entregadores'";
            }
        } catch (Exception $e) {
            // Coluna já existe ou erro
        }
    }
    
    // 4.1 Adicionar colunas na tabela usuarios (se não existirem)
    $colunasUsuario = [
        'bloqueado' => "ALTER TABLE usuarios ADD COLUMN bloqueado TINYINT(1) DEFAULT 0",
        'motivo_bloqueio' => "ALTER TABLE usuarios ADD COLUMN motivo_bloqueio TEXT NULL"
    ];
    
    foreach ($colunasUsuario as $coluna => $sql) {
        try {
            $stmt = $db->query("SHOW COLUMNS FROM usuarios LIKE '$coluna'");
            if ($stmt->rowCount() == 0) {
                $db->exec($sql);
                $mensagens[] = "✅ Coluna '$coluna' adicionada em 'usuarios'";
            }
        } catch (Exception $e) {
            // Coluna já existe ou erro
        }
    }
    
    // 4.2 Adicionar colunas na tabela entregas (se não existirem)
    $colunasEntregas = [
        'distancia' => "ALTER TABLE entregas ADD COLUMN distancia DECIMAL(8, 2) NULL COMMENT 'Distância em km'",
        'categoria_veiculo' => "ALTER TABLE entregas ADD COLUMN categoria_veiculo ENUM('moto', 'carro', 'van', 'caminhao') DEFAULT 'moto'",
        'peso_kg' => "ALTER TABLE entregas ADD COLUMN peso_kg DECIMAL(8, 2) NULL COMMENT 'Peso em kg'",
        'volume_m3' => "ALTER TABLE entregas ADD COLUMN volume_m3 DECIMAL(8, 3) NULL COMMENT 'Volume em m³'",
        'tipo_carga' => "ALTER TABLE entregas ADD COLUMN tipo_carga ENUM('documentos', 'eletronicos', 'roupas', 'alimentos', 'moveis', 'fragil', 'perecivel', 'outros') DEFAULT 'outros'",
        'dimensoes' => "ALTER TABLE entregas ADD COLUMN dimensoes VARCHAR(50) NULL COMMENT 'Dimensões aproximadas (ex: 30x20x10cm)'",
        'veiculo_sugerido' => "ALTER TABLE entregas ADD COLUMN veiculo_sugerido ENUM('moto', 'carro', 'van', 'caminhao') NULL COMMENT 'Veículo sugerido pelo sistema'",
        'cancelada_por' => "ALTER TABLE entregas ADD COLUMN cancelada_por ENUM('cliente', 'entregador', 'admin') NULL COMMENT 'Quem cancelou a entrega'",
        'motivo_cancelamento' => "ALTER TABLE entregas ADD COLUMN motivo_cancelamento VARCHAR(255) NULL COMMENT 'Motivo do cancelamento'",
        'data_cancelamento' => "ALTER TABLE entregas ADD COLUMN data_cancelamento DATETIME NULL COMMENT 'Data e hora do cancelamento'"
    ];
    
    foreach ($colunasEntregas as $coluna => $sql) {
        try {
            $stmt = $db->query("SHOW COLUMNS FROM entregas LIKE '$coluna'");
            if ($stmt->rowCount() == 0) {
                $db->exec($sql);
                $mensagens[] = "✅ Coluna '$coluna' adicionada em 'entregas'";
            }
        } catch (Exception $e) {
            // Coluna já existe ou erro
        }
    }
    
    // 5. Criar tabela de banners promocionais
    $db->exec("
        CREATE TABLE IF NOT EXISTS banners (
            id INT AUTO_INCREMENT PRIMARY KEY,
            titulo VARCHAR(100) NOT NULL,
            descricao VARCHAR(255),
            imagem VARCHAR(255),
            cor_fundo VARCHAR(20) DEFAULT '#1a73e8',
            cor_texto VARCHAR(20) DEFAULT '#ffffff',
            link VARCHAR(255),
            ordem INT DEFAULT 0,
            ativo TINYINT(1) DEFAULT 1,
            created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
        )
    ");
    $mensagens[] = "✅ Tabela 'banners' verificada/criada";
    
    // 5.1 Criar tabela de perfis de veículos
    $db->exec("
        CREATE TABLE IF NOT EXISTS perfis_veiculos (
            id INT AUTO_INCREMENT PRIMARY KEY,
            tipo ENUM('moto', 'carro', 'van', 'caminhao') NOT NULL,
            nome VARCHAR(50) NOT NULL,
            peso_max_kg DECIMAL(8, 2) NOT NULL COMMENT 'Peso máximo em kg',
            volume_max_m3 DECIMAL(8, 3) NOT NULL COMMENT 'Volume máximo em m³',
            dimensao_max_cm VARCHAR(50) COMMENT 'Dimensões máximas (ex: 100x80x60cm)',
            tipos_carga_aceitos TEXT COMMENT 'Tipos de carga aceitos (JSON)',
            icone VARCHAR(10) DEFAULT '🚗',
            descricao VARCHAR(255),
            ativo TINYINT(1) DEFAULT 1,
            created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
        )
    ");
    $mensagens[] = "✅ Tabela 'perfis_veiculos' verificada/criada";
    
    // 5.2 Inserir perfis padrão de veículos
    $perfisPadrao = [
        ['moto', 'Moto/Scooter', 15.00, 0.050, '40x30x25cm', '["documentos","eletronicos","roupas","outros"]', '🏍️', 'Ideal para documentos, pequenos pacotes e entregas rápidas'],
        ['carro', 'Carro', 50.00, 0.400, '80x60x40cm', '["documentos","eletronicos","roupas","alimentos","outros"]', '🚗', 'Perfeito para pacotes médios, compras e produtos eletrônicos'],
        ['van', 'Van/Utilitário', 500.00, 3.000, '200x120x100cm', '["eletronicos","roupas","alimentos","moveis","outros"]', '🚐', 'Ideal para móveis pequenos, mudanças e volumes maiores'],
        ['caminhao', 'Caminhão', 3000.00, 15.000, '400x200x200cm', '["moveis","outros"]', '🚛', 'Para mudanças completas, móveis grandes e cargas pesadas']
    ];
    
    $stmt = $db->prepare("INSERT IGNORE INTO perfis_veiculos (tipo, nome, peso_max_kg, volume_max_m3, dimensao_max_cm, tipos_carga_aceitos, icone, descricao) VALUES (?, ?, ?, ?, ?, ?, ?, ?)");
    foreach ($perfisPadrao as $perfil) {
        $stmt->execute($perfil);
    }
    $mensagens[] = "✅ Perfis padrão de veículos inseridos";
    
    // 5.3 Criar tabela de ganhos dos entregadores
    $db->exec("
        CREATE TABLE IF NOT EXISTS ganhos_entregadores (
            id INT AUTO_INCREMENT PRIMARY KEY,
            entregador_id INT NOT NULL,
            entrega_id INT NOT NULL,
            valor_bruto DECIMAL(10, 2) NOT NULL COMMENT 'Valor total da entrega',
            taxa_percentual DECIMAL(5, 2) NOT NULL COMMENT 'Taxa percentual aplicada (%)',
            taxa_fixa DECIMAL(10, 2) NOT NULL COMMENT 'Taxa fixa aplicada (R$)',
            valor_taxa DECIMAL(10, 2) NOT NULL COMMENT 'Valor total da taxa descontada',
            valor_liquido DECIMAL(10, 2) NOT NULL COMMENT 'Valor que o entregador recebe',
            data_pagamento DATETIME DEFAULT CURRENT_TIMESTAMP,
            FOREIGN KEY (entregador_id) REFERENCES entregadores(id),
            FOREIGN KEY (entrega_id) REFERENCES entregas(id),
            UNIQUE KEY unique_entrega (entrega_id)
        )
    ");
    $mensagens[] = "✅ Tabela 'ganhos_entregadores' verificada/criada";
    
    // 6. Inserir configurações padrão (se não existirem)
    $configsPadrao = [
        ['nome_empresa', 'Já EntreGo', 'Nome da empresa exibido nos apps'],
        ['email_empresa', 'contato@jaentrego.com', 'Email de contato da empresa'],
        ['telefone_empresa', '', 'Telefone de contato da empresa'],
        ['endereco_empresa', '', 'Endereço da empresa'],
        ['taxa_entregador_percentual', '15', 'Taxa percentual cobrada do entregador por entrega (%)'],
        ['taxa_entregador_fixa', '0', 'Taxa fixa cobrada do entregador por entrega (R$)'],
        ['valor_minimo_entrega', '10', 'Valor mínimo de uma entrega (R$)'],
        ['raio_maximo_km', '50', 'Raio máximo de entrega em km'],
        // Preços por categoria - Moto
        ['preco_base_moto', '8', 'Taxa base para entregas de moto (R$)'],
        ['preco_km_moto', '2.00', 'Valor por km para moto (R$)'],
        // Preços por categoria - Carro
        ['preco_base_carro', '12', 'Taxa base para entregas de carro (R$)'],
        ['preco_km_carro', '2.50', 'Valor por km para carro (R$)'],
        // Preços por categoria - Van
        ['preco_base_van', '20', 'Taxa base para entregas de van (R$)'],
        ['preco_km_van', '3.50', 'Valor por km para van (R$)'],
        // Preços por categoria - Caminhão
        ['preco_base_caminhao', '35', 'Taxa base para entregas de caminhão (R$)'],
        ['preco_km_caminhao', '5.00', 'Valor por km para caminhão (R$)']
    ];
    
    $stmt = $db->prepare("INSERT IGNORE INTO configuracoes (chave, valor, descricao) VALUES (?, ?, ?)");
    foreach ($configsPadrao as $cfg) {
        $stmt->execute($cfg);
    }
    $mensagens[] = "✅ Configurações padrão verificadas/inseridas";
    
    // 6. Verificar se admin existe, se não criar
    $stmt = $db->query("SELECT COUNT(*) as total FROM admins");
    $total = $stmt->fetch()['total'];
    if ($total == 0) {
        $senhaHash = password_hash('admin123', PASSWORD_DEFAULT);
        $db->exec("INSERT INTO admins (nome, email, senha) VALUES ('Admin', 'admin@entregas.com', '$senhaHash')");
        $mensagens[] = "✅ Admin padrão criado (admin@entregas.com / admin123)";
    }
    
    $sucesso = true;
    
} catch (PDOException $e) {
    $erros[] = "❌ Erro: " . $e->getMessage();
    $sucesso = false;
}
?>
<!DOCTYPE html>
<html lang="pt-BR">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Atualização do Banco - Já EntreGo</title>
    <style>
        * { margin: 0; padding: 0; box-sizing: border-box; }
        body {
            font-family: 'Segoe UI', Tahoma, Geneva, Verdana, sans-serif;
            background: linear-gradient(135deg, #1a73e8 0%, #0d47a1 100%);
            min-height: 100vh;
            display: flex;
            align-items: center;
            justify-content: center;
            padding: 20px;
        }
        .container {
            background: white;
            border-radius: 16px;
            padding: 40px;
            width: 100%;
            max-width: 600px;
            box-shadow: 0 10px 40px rgba(0,0,0,0.2);
        }
        .logo { text-align: center; font-size: 60px; margin-bottom: 10px; }
        h1 { text-align: center; color: #1a73e8; margin-bottom: 10px; }
        .subtitle { text-align: center; color: #666; margin-bottom: 30px; }
        .resultado {
            background: #f5f5f5;
            border-radius: 12px;
            padding: 20px;
            margin-bottom: 20px;
        }
        .resultado h3 {
            margin-bottom: 15px;
            color: #333;
        }
        .item {
            padding: 10px 0;
            border-bottom: 1px solid #e0e0e0;
            font-size: 14px;
        }
        .item:last-child { border-bottom: none; }
        .sucesso { color: #2e7d32; }
        .erro { color: #c62828; }
        .alert {
            padding: 16px;
            border-radius: 8px;
            margin-bottom: 20px;
            text-align: center;
        }
        .alert-success { background: #e8f5e9; color: #2e7d32; }
        .alert-danger { background: #ffebee; color: #c62828; }
        .links {
            display: flex;
            gap: 10px;
            flex-wrap: wrap;
        }
        .links a {
            flex: 1;
            min-width: 150px;
            padding: 12px;
            background: #1a73e8;
            color: white;
            text-decoration: none;
            border-radius: 8px;
            text-align: center;
            font-weight: 500;
        }
        .links a:hover { background: #1557b0; }
        .warning {
            background: #fff3e0;
            border: 1px solid #ffcc80;
            color: #e65100;
            padding: 12px;
            border-radius: 8px;
            margin-top: 20px;
            font-size: 14px;
        }
    </style>
</head>
<body>
    <div class="container">
        <div class="logo"><?= $sucesso ? '✅' : '❌' ?></div>
        <h1>Atualização do Banco</h1>
        <p class="subtitle">Já EntreGo - Sistema de Entregas</p>
        
        <?php if ($sucesso): ?>
            <div class="alert alert-success">
                <strong>Atualização concluída com sucesso!</strong>
            </div>
        <?php else: ?>
            <div class="alert alert-danger">
                <strong>Ocorreram erros durante a atualização</strong>
            </div>
        <?php endif; ?>
        
        <div class="resultado">
            <h3>📋 Log de Execução</h3>
            <?php foreach ($mensagens as $msg): ?>
                <div class="item sucesso"><?= $msg ?></div>
            <?php endforeach; ?>
            <?php foreach ($erros as $err): ?>
                <div class="item erro"><?= $err ?></div>
            <?php endforeach; ?>
        </div>
        
        <?php if ($sucesso): ?>
            <div class="links">
                <a href="index.php">🛒 App Cliente</a>
                <a href="entregador/">🏍️ App Entregador</a>
                <a href="admin/">⚙️ Painel Admin</a>
            </div>
            
            <div class="warning">
                <strong>⚠️ Importante:</strong> Por segurança, delete este arquivo (atualizar_banco.php) após a atualização!
            </div>
        <?php endif; ?>
    </div>
</body>
</html>
