<?php
/**
 * Sistema Inteligente de Seleção de Veículos
 * Analisa as características da carga e sugere o veículo ideal
 */

class VeiculoInteligente {
    private $db;
    private $perfisVeiculos = [];
    
    public function __construct($database) {
        $this->db = $database;
        $this->carregarPerfis();
    }
    
    /**
     * Carrega os perfis de veículos do banco
     */
    private function carregarPerfis() {
        $stmt = $this->db->query("SELECT * FROM perfis_veiculos WHERE ativo = 1 ORDER BY peso_max_kg ASC");
        $this->perfisVeiculos = $stmt->fetchAll(PDO::FETCH_ASSOC);
    }
    
    /**
     * Analisa a carga e sugere o veículo ideal
     * 
     * @param array $dadosCarga - Array com: peso_kg, volume_m3, tipo_carga, dimensoes
     * @return array - Veículo sugerido e alternativas
     */
    public function analisarCarga($dadosCarga) {
        $peso = floatval($dadosCarga['peso_kg'] ?? 0);
        $volume = floatval($dadosCarga['volume_m3'] ?? 0);
        $tipoCarga = $dadosCarga['tipo_carga'] ?? 'outros';
        $dimensoes = $dadosCarga['dimensoes'] ?? '';
        
        $veiculosCompativeis = [];
        $pontuacoes = [];
        
        foreach ($this->perfisVeiculos as $perfil) {
            $compativel = $this->verificarCompatibilidade($perfil, $peso, $volume, $tipoCarga);
            
            if ($compativel['compativel']) {
                $veiculosCompativeis[] = $perfil;
                $pontuacoes[$perfil['tipo']] = $this->calcularPontuacao($perfil, $peso, $volume, $tipoCarga);
            }
        }
        
        if (empty($veiculosCompativeis)) {
            // Se nenhum veículo for compatível, sugerir o maior
            $veiculoSugerido = end($this->perfisVeiculos);
            return [
                'sugerido' => $veiculoSugerido,
                'motivo' => 'Carga excede limites padrão - veículo maior disponível',
                'alternativas' => [],
                'compativel' => false
            ];
        }
        
        // Ordenar por pontuação (maior pontuação = mais adequado)
        arsort($pontuacoes);
        $tipoSugerido = array_key_first($pontuacoes);
        
        $veiculoSugerido = null;
        foreach ($veiculosCompativeis as $veiculo) {
            if ($veiculo['tipo'] === $tipoSugerido) {
                $veiculoSugerido = $veiculo;
                break;
            }
        }
        
        // Remover o sugerido das alternativas
        $alternativas = array_filter($veiculosCompativeis, function($v) use ($tipoSugerido) {
            return $v['tipo'] !== $tipoSugerido;
        });
        
        return [
            'sugerido' => $veiculoSugerido,
            'motivo' => $this->gerarMotivo($veiculoSugerido, $peso, $volume, $tipoCarga),
            'alternativas' => array_values($alternativas),
            'compativel' => true
        ];
    }
    
    /**
     * Verifica se um veículo é compatível com a carga
     */
    private function verificarCompatibilidade($perfil, $peso, $volume, $tipoCarga) {
        // Verificar peso
        if ($peso > $perfil['peso_max_kg']) {
            return ['compativel' => false, 'motivo' => 'Peso excede limite'];
        }
        
        // Verificar volume
        if ($volume > $perfil['volume_max_m3']) {
            return ['compativel' => false, 'motivo' => 'Volume excede limite'];
        }
        
        // Verificar tipo de carga
        $tiposAceitos = json_decode($perfil['tipos_carga_aceitos'], true) ?? [];
        if (!in_array($tipoCarga, $tiposAceitos) && !in_array('outros', $tiposAceitos)) {
            return ['compativel' => false, 'motivo' => 'Tipo de carga não aceito'];
        }
        
        return ['compativel' => true, 'motivo' => 'Compatível'];
    }
    
    /**
     * Calcula pontuação de adequação (0-100)
     * Maior pontuação = mais adequado
     */
    private function calcularPontuacao($perfil, $peso, $volume, $tipoCarga) {
        $pontuacao = 0;
        
        // Pontuação por eficiência de peso (usar 60-80% da capacidade é ideal)
        $percentualPeso = ($peso / $perfil['peso_max_kg']) * 100;
        if ($percentualPeso >= 60 && $percentualPeso <= 80) {
            $pontuacao += 40; // Faixa ideal
        } elseif ($percentualPeso >= 40 && $percentualPeso < 60) {
            $pontuacao += 30; // Boa faixa
        } elseif ($percentualPeso >= 20 && $percentualPeso < 40) {
            $pontuacao += 20; // Aceitável
        } else {
            $pontuacao += 10; // Subutilizado ou no limite
        }
        
        // Pontuação por eficiência de volume
        $percentualVolume = ($volume / $perfil['volume_max_m3']) * 100;
        if ($percentualVolume >= 50 && $percentualVolume <= 80) {
            $pontuacao += 30;
        } elseif ($percentualVolume >= 30 && $percentualVolume < 50) {
            $pontuacao += 25;
        } elseif ($percentualVolume >= 10 && $percentualVolume < 30) {
            $pontuacao += 15;
        } else {
            $pontuacao += 5;
        }
        
        // Pontuação por tipo de carga específico
        $tiposAceitos = json_decode($perfil['tipos_carga_aceitos'], true) ?? [];
        if (in_array($tipoCarga, $tiposAceitos)) {
            $pontuacao += 20;
        }
        
        // Bonificação por economia (veículos menores são mais econômicos)
        $bonusEconomia = [
            'moto' => 10,
            'carro' => 7,
            'van' => 4,
            'caminhao' => 0
        ];
        $pontuacao += $bonusEconomia[$perfil['tipo']] ?? 0;
        
        return $pontuacao;
    }
    
    /**
     * Gera motivo da sugestão
     */
    private function gerarMotivo($perfil, $peso, $volume, $tipoCarga) {
        $motivos = [];
        
        if ($peso > 0) {
            $percentualPeso = ($peso / $perfil['peso_max_kg']) * 100;
            if ($percentualPeso >= 60 && $percentualPeso <= 80) {
                $motivos[] = "peso ideal para este veículo";
            } elseif ($percentualPeso < 30) {
                $motivos[] = "carga leve, veículo econômico";
            }
        }
        
        if ($volume > 0) {
            $percentualVolume = ($volume / $perfil['volume_max_m3']) * 100;
            if ($percentualVolume >= 50) {
                $motivos[] = "volume adequado";
            }
        }
        
        // Motivos específicos por tipo de carga
        $motivosCarga = [
            'documentos' => 'ideal para documentos e papéis',
            'eletronicos' => 'adequado para eletrônicos',
            'roupas' => 'perfeito para roupas e tecidos',
            'alimentos' => 'apropriado para alimentos',
            'moveis' => 'ideal para móveis e objetos grandes',
            'fragil' => 'cuidado especial com itens frágeis',
            'perecivel' => 'transporte rápido para perecíveis'
        ];
        
        if (isset($motivosCarga[$tipoCarga])) {
            $motivos[] = $motivosCarga[$tipoCarga];
        }
        
        if (empty($motivos)) {
            return "Veículo mais adequado para suas necessidades";
        }
        
        return "Sugerido por: " . implode(', ', $motivos);
    }
    
    /**
     * Estima peso baseado no tipo de carga e dimensões
     */
    public static function estimarPeso($tipoCarga, $dimensoes = '') {
        // Densidades aproximadas por tipo de carga (kg/m³)
        $densidades = [
            'documentos' => 200,    // Papéis são leves
            'eletronicos' => 800,   // Eletrônicos são densos
            'roupas' => 300,        // Roupas são leves
            'alimentos' => 600,     // Varia muito, média
            'moveis' => 400,        // Madeira/metal
            'fragil' => 500,        // Vidro, cerâmica
            'perecivel' => 600,     // Similar a alimentos
            'outros' => 500         // Média geral
        ];
        
        $densidade = $densidades[$tipoCarga] ?? 500;
        
        // Se tiver dimensões, calcular volume e estimar peso
        if ($dimensoes && preg_match('/(\d+)x(\d+)x(\d+)/', $dimensoes, $matches)) {
            $comprimento = floatval($matches[1]) / 100; // cm para m
            $largura = floatval($matches[2]) / 100;
            $altura = floatval($matches[3]) / 100;
            $volume = $comprimento * $largura * $altura;
            
            return $volume * $densidade;
        }
        
        // Estimativas padrão por tipo
        $pesosPadrao = [
            'documentos' => 0.5,
            'eletronicos' => 2.0,
            'roupas' => 1.0,
            'alimentos' => 3.0,
            'moveis' => 20.0,
            'fragil' => 2.0,
            'perecivel' => 3.0,
            'outros' => 5.0
        ];
        
        return $pesosPadrao[$tipoCarga] ?? 5.0;
    }
    
    /**
     * Estima volume baseado no tipo de carga e dimensões
     */
    public static function estimarVolume($tipoCarga, $dimensoes = '') {
        // Se tiver dimensões específicas
        if ($dimensoes && preg_match('/(\d+)x(\d+)x(\d+)/', $dimensoes, $matches)) {
            $comprimento = floatval($matches[1]) / 100; // cm para m
            $largura = floatval($matches[2]) / 100;
            $altura = floatval($matches[3]) / 100;
            return $comprimento * $largura * $altura;
        }
        
        // Volumes padrão por tipo (m³)
        $volumesPadrao = [
            'documentos' => 0.002,  // Envelope A4
            'eletronicos' => 0.010, // Caixa pequena
            'roupas' => 0.020,      // Sacola
            'alimentos' => 0.015,   // Caixa média
            'moveis' => 0.200,      // Móvel pequeno
            'fragil' => 0.008,      // Embalagem protegida
            'perecivel' => 0.015,   // Similar a alimentos
            'outros' => 0.025       // Caixa padrão
        ];
        
        return $volumesPadrao[$tipoCarga] ?? 0.025;
    }
    
    /**
     * Retorna todos os perfis de veículos
     */
    public function getPerfisVeiculos() {
        return $this->perfisVeiculos;
    }
}