-- Banco de dados da plataforma de entregas
CREATE DATABASE IF NOT EXISTS entregas_db CHARACTER SET utf8mb4 COLLATE utf8mb4_unicode_ci;
USE entregas_db;

-- Tabela de usuários (clientes)
CREATE TABLE usuarios (
    id INT AUTO_INCREMENT PRIMARY KEY,
    nome VARCHAR(100) NOT NULL,
    email VARCHAR(100) UNIQUE NOT NULL,
    senha VARCHAR(255) NOT NULL,
    telefone VARCHAR(20),
    bloqueado TINYINT(1) DEFAULT 0,
    motivo_bloqueio TEXT,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
);

-- Tabela de entregadores
CREATE TABLE entregadores (
    id INT AUTO_INCREMENT PRIMARY KEY,
    nome VARCHAR(100) NOT NULL,
    email VARCHAR(100) UNIQUE NOT NULL,
    senha VARCHAR(255) NOT NULL,
    telefone VARCHAR(20),
    cpf VARCHAR(14),
    endereco VARCHAR(255),
    
    -- Tipo de veículo
    veiculo ENUM('bicicleta', 'moto', 'carro', 'van', 'caminhao') DEFAULT 'moto',
    
    -- Dados do veículo (para moto, carro, van, caminhão)
    placa VARCHAR(10),
    veiculo_marca VARCHAR(50),
    veiculo_modelo VARCHAR(50),
    veiculo_ano VARCHAR(4),
    veiculo_cor VARCHAR(30),
    
    -- Documentos (caminhos dos arquivos)
    doc_identidade VARCHAR(255),
    doc_cpf VARCHAR(255),
    doc_cnh VARCHAR(255),
    doc_veiculo VARCHAR(255),
    doc_endereco VARCHAR(255),
    
    -- Status de aprovação
    aprovado ENUM('pendente', 'aprovado', 'rejeitado') DEFAULT 'pendente',
    motivo_rejeicao TEXT,
    data_aprovacao DATETIME,
    
    -- Status online
    status ENUM('disponivel', 'ocupado', 'offline') DEFAULT 'offline',
    latitude DECIMAL(10, 8),
    longitude DECIMAL(11, 8),
    bloqueado TINYINT(1) DEFAULT 0,
    motivo_bloqueio TEXT,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
);

-- Tabela de administradores
CREATE TABLE admins (
    id INT AUTO_INCREMENT PRIMARY KEY,
    nome VARCHAR(100) NOT NULL,
    email VARCHAR(100) UNIQUE NOT NULL,
    senha VARCHAR(255) NOT NULL,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
);

-- Tabela de entregas
CREATE TABLE entregas (
    id INT AUTO_INCREMENT PRIMARY KEY,
    usuario_id INT NOT NULL,
    entregador_id INT,
    origem_endereco VARCHAR(255) NOT NULL,
    origem_lat DECIMAL(10, 8),
    origem_lng DECIMAL(11, 8),
    destino_endereco VARCHAR(255) NOT NULL,
    destino_lat DECIMAL(10, 8),
    destino_lng DECIMAL(11, 8),
    descricao TEXT,
    valor DECIMAL(10, 2) NOT NULL,
    status ENUM('pendente', 'aceita', 'coletada', 'entregue', 'cancelada') DEFAULT 'pendente',
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    FOREIGN KEY (usuario_id) REFERENCES usuarios(id),
    FOREIGN KEY (entregador_id) REFERENCES entregadores(id)
);

-- Tabela de notificações para entregadores
CREATE TABLE notificacoes (
    id INT AUTO_INCREMENT PRIMARY KEY,
    entregador_id INT,
    entrega_id INT,
    mensagem VARCHAR(255),
    lida TINYINT(1) DEFAULT 0,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    FOREIGN KEY (entregador_id) REFERENCES entregadores(id),
    FOREIGN KEY (entrega_id) REFERENCES entregas(id)
);

-- Tabela de notificações para clientes
CREATE TABLE notificacoes_cliente (
    id INT AUTO_INCREMENT PRIMARY KEY,
    usuario_id INT NOT NULL,
    entrega_id INT NOT NULL,
    tipo ENUM('aceita', 'coletada', 'entregue', 'cancelada') NOT NULL,
    mensagem VARCHAR(255),
    lida TINYINT(1) DEFAULT 0,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    FOREIGN KEY (usuario_id) REFERENCES usuarios(id),
    FOREIGN KEY (entrega_id) REFERENCES entregas(id)
);

-- Tabela de configurações do sistema
CREATE TABLE configuracoes (
    id INT AUTO_INCREMENT PRIMARY KEY,
    chave VARCHAR(50) UNIQUE NOT NULL,
    valor TEXT,
    descricao VARCHAR(255),
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP
);

-- Tabela de empresas parceiras
CREATE TABLE empresas_parceiras (
    id INT AUTO_INCREMENT PRIMARY KEY,
    nome_empresa VARCHAR(100) NOT NULL,
    cnpj VARCHAR(20) UNIQUE,
    email VARCHAR(100) NOT NULL,
    telefone VARCHAR(20),
    endereco VARCHAR(255),
    responsavel VARCHAR(100),
    status ENUM('pendente', 'aprovada', 'rejeitada') DEFAULT 'pendente',
    observacoes TEXT,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
);

-- Inserir configurações padrão
INSERT INTO configuracoes (chave, valor, descricao) VALUES 
('nome_empresa', 'Já EntreGo', 'Nome da empresa exibido nos apps'),
('email_empresa', 'contato@jaentrego.com', 'Email de contato da empresa'),
('telefone_empresa', '', 'Telefone de contato da empresa'),
('endereco_empresa', '', 'Endereço da empresa'),
('taxa_entregador_percentual', '15', 'Taxa percentual cobrada do entregador por entrega (%)'),
('taxa_entregador_fixa', '0', 'Taxa fixa cobrada do entregador por entrega (R$)'),
('valor_minimo_entrega', '10', 'Valor mínimo de uma entrega (R$)'),
('valor_por_km', '2.50', 'Valor cobrado por km rodado (R$)');

-- Inserir admin padrão (senha: admin123)
INSERT INTO admins (nome, email, senha) VALUES ('Admin', 'admin@entregas.com', '$2y$10$E8rKzL5xN3qW7vY9uI1oAeZcXbMnOpQrStUvWxYzAbCdEfGhIjKlM');
