<?php
require_once '../../config/database.php';
require_once '../../config/auth.php';
require_once '../../config/helpers.php';

header('Content-Type: application/json');

if (!isLoggedIn('entregador')) {
    echo json_encode(['success' => false, 'message' => 'Não autorizado']);
    exit;
}

$entregaId = intval($_POST['entrega_id'] ?? 0);
$status = $_POST['status'] ?? '';

if (!$entregaId || !in_array($status, ['coletada', 'entregue'])) {
    echo json_encode(['success' => false, 'message' => 'Dados inválidos']);
    exit;
}

$db = Database::getInstance()->getConnection();
$entregadorId = getCurrentUserId('entregador');

// Verificar se a entrega pertence ao entregador
$stmt = $db->prepare("SELECT id, status, valor, usuario_id FROM entregas WHERE id = ? AND entregador_id = ?");
$stmt->execute([$entregaId, $entregadorId]);
$entrega = $stmt->fetch(PDO::FETCH_ASSOC);

if (!$entrega) {
    echo json_encode(['success' => false, 'message' => 'Entrega não encontrada']);
    exit;
}

// Validar transição de status
$transicaoValida = ($entrega['status'] === 'aceita' && $status === 'coletada') ||
                   ($entrega['status'] === 'coletada' && $status === 'entregue');

if (!$transicaoValida) {
    echo json_encode(['success' => false, 'message' => 'Transição de status inválida']);
    exit;
}

try {
    $db->beginTransaction();
    
    // Atualizar status
    $stmt = $db->prepare("UPDATE entregas SET status = ? WHERE id = ?");
    $result = $stmt->execute([$status, $entregaId]);
    
    if (!$result) {
        throw new Exception('Erro ao atualizar status da entrega');
    }
    
    // Buscar dados do entregador para notificação
    $stmt = $db->prepare("SELECT nome FROM entregadores WHERE id = ?");
    $stmt->execute([$entregadorId]);
    $entregadorNome = $stmt->fetch(PDO::FETCH_ASSOC)['nome'];
    
    // Criar notificação para o cliente
    if ($status === 'coletada') {
        $msg = $entregadorNome . " coletou seu pacote e está a caminho!";
    } else {
        $msg = "Sua entrega foi realizada com sucesso! 🎉";
    }
    $stmt = $db->prepare("INSERT INTO notificacoes_cliente (usuario_id, entrega_id, tipo, mensagem) VALUES (?, ?, ?, ?)");
    $stmt->execute([$entrega['usuario_id'], $entregaId, $status, $msg]);
    
    // Se entregue, processar pagamento e liberar entregador
    if ($status === 'entregue') {
        // Calcular ganhos do entregador
        $valorBruto = floatval($entrega['valor']);
        $taxaPercentual = floatval(getConfig('taxa_entregador_percentual', 15));
        $taxaFixa = floatval(getConfig('taxa_entregador_fixa', 0));
        
        $valorTaxa = ($valorBruto * ($taxaPercentual / 100)) + $taxaFixa;
        $valorLiquido = max(0, $valorBruto - $valorTaxa);
        
        // Verificar se já não foi processado
        $stmt = $db->prepare("SELECT id FROM ganhos_entregadores WHERE entrega_id = ?");
        $stmt->execute([$entregaId]);
        
        if (!$stmt->fetch()) {
            // Registrar ganho do entregador
            $stmt = $db->prepare("
                INSERT INTO ganhos_entregadores 
                (entregador_id, entrega_id, valor_bruto, taxa_percentual, taxa_fixa, valor_taxa, valor_liquido) 
                VALUES (?, ?, ?, ?, ?, ?, ?)
            ");
            $stmt->execute([
                $entregadorId, 
                $entregaId, 
                $valorBruto, 
                $taxaPercentual, 
                $taxaFixa, 
                $valorTaxa, 
                $valorLiquido
            ]);
        }
        
        // Liberar entregador
        $stmt = $db->prepare("UPDATE entregadores SET status = 'disponivel' WHERE id = ?");
        $stmt->execute([$entregadorId]);
    }
    
    $db->commit();
    
    $response = ['success' => true];
    
    // Adicionar informações de ganho se for entrega concluída
    if ($status === 'entregue') {
        $response['ganho'] = [
            'valor_bruto' => $valorBruto,
            'valor_taxa' => $valorTaxa,
            'valor_liquido' => $valorLiquido,
            'taxa_percentual' => $taxaPercentual
        ];
    }
    
    echo json_encode($response);
    
} catch (Exception $e) {
    $db->rollBack();
    echo json_encode(['success' => false, 'message' => 'Erro ao processar: ' . $e->getMessage()]);
}
