<?php
require_once '../config/database.php';
require_once '../config/auth.php';
require_once '../config/email.php';

$erro = '';
$sucesso = '';

// Função para validar placa (formato antigo e Mercosul)
function validarPlaca($placa) {
    $placa = strtoupper(preg_replace('/[^A-Za-z0-9]/', '', $placa));
    // Formato antigo: ABC1234 ou ABC-1234
    // Formato Mercosul: ABC1D23
    return preg_match('/^[A-Z]{3}[0-9]{4}$/', $placa) || preg_match('/^[A-Z]{3}[0-9][A-Z][0-9]{2}$/', $placa);
}

// Função para fazer upload de arquivo
function uploadDocumento($arquivo, $pasta, $prefixo) {
    $extensoesPermitidas = ['jpg', 'jpeg', 'png', 'pdf'];
    $tamanhoMaximo = 5 * 1024 * 1024; // 5MB
    
    if ($arquivo['error'] !== UPLOAD_ERR_OK) {
        return ['erro' => 'Erro no upload do arquivo'];
    }
    
    if ($arquivo['size'] > $tamanhoMaximo) {
        return ['erro' => 'Arquivo muito grande (máximo 5MB)'];
    }
    
    $extensao = strtolower(pathinfo($arquivo['name'], PATHINFO_EXTENSION));
    if (!in_array($extensao, $extensoesPermitidas)) {
        return ['erro' => 'Formato não permitido. Use: JPG, PNG ou PDF'];
    }
    
    // Criar pasta se não existir
    $caminhoCompleto = __DIR__ . '/../uploads/' . $pasta;
    if (!is_dir($caminhoCompleto)) {
        mkdir($caminhoCompleto, 0755, true);
    }
    
    // Gerar nome único
    $nomeArquivo = $prefixo . '_' . uniqid() . '.' . $extensao;
    $destino = $caminhoCompleto . '/' . $nomeArquivo;
    
    if (move_uploaded_file($arquivo['tmp_name'], $destino)) {
        return ['sucesso' => 'uploads/' . $pasta . '/' . $nomeArquivo];
    }
    
    return ['erro' => 'Erro ao salvar arquivo'];
}

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    // Dados pessoais
    $nome = trim($_POST['nome'] ?? '');
    $email = trim($_POST['email'] ?? '');
    $telefone = trim($_POST['telefone'] ?? '');
    $cpf = trim($_POST['cpf'] ?? '');
    $endereco = trim($_POST['endereco'] ?? '');
    $senha = $_POST['senha'] ?? '';
    $confirmar = $_POST['confirmar'] ?? '';
    
    // Tipo de veículo
    $veiculo = $_POST['veiculo'] ?? 'moto';
    
    // Dados do veículo (se não for bicicleta)
    $placa = trim($_POST['placa'] ?? '');
    $marca = trim($_POST['marca'] ?? '');
    $modelo = trim($_POST['modelo'] ?? '');
    $ano = trim($_POST['ano'] ?? '');
    $cor = trim($_POST['cor'] ?? '');
    
    // Validações básicas
    if (empty($nome) || empty($email) || empty($senha) || empty($telefone) || empty($cpf) || empty($endereco)) {
        $erro = 'Preencha todos os campos obrigatórios';
    } elseif ($senha !== $confirmar) {
        $erro = 'As senhas não conferem';
    } elseif (strlen($senha) < 6) {
        $erro = 'A senha deve ter pelo menos 6 caracteres';
    } elseif (!filter_var($email, FILTER_VALIDATE_EMAIL)) {
        $erro = 'Email inválido';
    } elseif ($veiculo !== 'bicicleta' && empty($placa)) {
        $erro = 'Informe a placa do veículo';
    } elseif ($veiculo !== 'bicicleta' && !validarPlaca($placa)) {
        $erro = 'Placa inválida. Use formato ABC-1234 ou ABC1D23 (Mercosul)';
    } else {
        // Validar documentos obrigatórios
        $docIdentidade = $_FILES['doc_identidade'] ?? null;
        $docCpf = $_FILES['doc_cpf'] ?? null;
        
        if (!$docIdentidade || $docIdentidade['error'] === UPLOAD_ERR_NO_FILE) {
            $erro = 'Envie o documento de identidade (RG)';
        } elseif (!$docCpf || $docCpf['error'] === UPLOAD_ERR_NO_FILE) {
            $erro = 'Envie o documento do CPF';
        } else {
            // Para veículos motorizados, exigir CNH e documento do veículo
            if ($veiculo !== 'bicicleta') {
                $docCnh = $_FILES['doc_cnh'] ?? null;
                $docVeiculo = $_FILES['doc_veiculo'] ?? null;
                
                if (!$docCnh || $docCnh['error'] === UPLOAD_ERR_NO_FILE) {
                    $erro = 'Envie a CNH';
                } elseif (!$docVeiculo || $docVeiculo['error'] === UPLOAD_ERR_NO_FILE) {
                    $erro = 'Envie o documento do veículo (CRLV)';
                }
            }
        }
    }
    
    if (empty($erro)) {
        $db = Database::getInstance()->getConnection();
        
        // Verificar se email já existe
        $stmt = $db->prepare("SELECT id FROM entregadores WHERE email = ?");
        $stmt->execute([$email]);
        
        if ($stmt->fetch()) {
            $erro = 'Este email já está cadastrado';
        } else {
            // Fazer upload dos documentos
            $uploads = [];
            $prefixo = preg_replace('/[^a-z0-9]/', '', strtolower($nome));
            
            // Upload identidade
            $result = uploadDocumento($_FILES['doc_identidade'], 'documentos', $prefixo . '_rg');
            if (isset($result['erro'])) {
                $erro = 'Identidade: ' . $result['erro'];
            } else {
                $uploads['identidade'] = $result['sucesso'];
            }
            
            // Upload CPF
            if (empty($erro)) {
                $result = uploadDocumento($_FILES['doc_cpf'], 'documentos', $prefixo . '_cpf');
                if (isset($result['erro'])) {
                    $erro = 'CPF: ' . $result['erro'];
                } else {
                    $uploads['cpf'] = $result['sucesso'];
                }
            }
            
            // Upload comprovante de endereço (opcional)
            if (empty($erro) && isset($_FILES['doc_endereco']) && $_FILES['doc_endereco']['error'] !== UPLOAD_ERR_NO_FILE) {
                $result = uploadDocumento($_FILES['doc_endereco'], 'documentos', $prefixo . '_endereco');
                if (isset($result['erro'])) {
                    $erro = 'Comprovante endereço: ' . $result['erro'];
                } else {
                    $uploads['endereco'] = $result['sucesso'];
                }
            }
            
            // Upload CNH e documento do veículo (se não for bicicleta)
            if (empty($erro) && $veiculo !== 'bicicleta') {
                $result = uploadDocumento($_FILES['doc_cnh'], 'documentos', $prefixo . '_cnh');
                if (isset($result['erro'])) {
                    $erro = 'CNH: ' . $result['erro'];
                } else {
                    $uploads['cnh'] = $result['sucesso'];
                }
                
                if (empty($erro)) {
                    $result = uploadDocumento($_FILES['doc_veiculo'], 'documentos', $prefixo . '_veiculo');
                    if (isset($result['erro'])) {
                        $erro = 'Documento veículo: ' . $result['erro'];
                    } else {
                        $uploads['veiculo'] = $result['sucesso'];
                    }
                }
            }
            
            // Se todos uploads OK, inserir no banco
            if (empty($erro)) {
                $senhaHash = password_hash($senha, PASSWORD_DEFAULT);
                $placaFormatada = $veiculo !== 'bicicleta' ? strtoupper(preg_replace('/[^A-Za-z0-9]/', '', $placa)) : null;
                
                $stmt = $db->prepare("
                    INSERT INTO entregadores (
                        nome, email, senha, telefone, cpf, endereco,
                        veiculo, placa, veiculo_marca, veiculo_modelo, veiculo_ano, veiculo_cor,
                        doc_identidade, doc_cpf, doc_cnh, doc_veiculo, doc_endereco,
                        aprovado
                    ) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, 'pendente')
                ");
                
                $result = $stmt->execute([
                    $nome, $email, $senhaHash, $telefone, $cpf, $endereco,
                    $veiculo, $placaFormatada, 
                    $veiculo !== 'bicicleta' ? $marca : null,
                    $veiculo !== 'bicicleta' ? $modelo : null,
                    $veiculo !== 'bicicleta' ? $ano : null,
                    $veiculo !== 'bicicleta' ? $cor : null,
                    $uploads['identidade'] ?? null,
                    $uploads['cpf'] ?? null,
                    $uploads['cnh'] ?? null,
                    $uploads['veiculo'] ?? null,
                    $uploads['endereco'] ?? null
                ]);
                
                if ($result) {
                    // Enviar email de confirmação
                    enviarEmailCadastroEntregador($email, $nome);
                    $sucesso = true;
                } else {
                    $erro = 'Erro ao cadastrar. Tente novamente.';
                }
            }
        }
    }
}
?>
<!DOCTYPE html>
<html lang="pt-BR">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Cadastro Entregador - Já EntreGo</title>
    <link rel="stylesheet" href="../assets/css/style.css">
    <link rel="stylesheet" href="../assets/css/entregador.css">
    <style>
        .cadastro-container {
            min-height: 100vh;
            background: linear-gradient(135deg, #0d47a1 0%, #002171 100%);
            padding: 20px;
        }
        .cadastro-box {
            background: white;
            max-width: 600px;
            margin: 0 auto;
            border-radius: 16px;
            padding: 30px;
            box-shadow: 0 10px 40px rgba(0,0,0,0.3);
        }
        .cadastro-box h2 {
            color: #0d47a1;
            text-align: center;
            margin-bottom: 10px;
        }
        .cadastro-box .subtitle {
            text-align: center;
            color: #666;
            margin-bottom: 25px;
        }
        .section-title {
            font-size: 14px;
            color: #0d47a1;
            text-transform: uppercase;
            margin: 25px 0 15px;
            padding-bottom: 8px;
            border-bottom: 2px solid #e0e0e0;
        }
        .form-row {
            display: flex;
            gap: 15px;
        }
        .form-row .form-group {
            flex: 1;
        }
        .file-input {
            border: 2px dashed #ccc;
            padding: 15px;
            border-radius: 8px;
            text-align: center;
            cursor: pointer;
            transition: all 0.3s;
        }
        .file-input:hover {
            border-color: #0d47a1;
            background: #f5f5f5;
        }
        .file-input input {
            display: none;
        }
        .file-input .icon {
            font-size: 30px;
            margin-bottom: 5px;
        }
        .file-input .text {
            font-size: 12px;
            color: #666;
        }
        .file-input.selected {
            border-color: #4caf50;
            background: #e8f5e9;
        }
        .veiculo-options {
            display: flex;
            gap: 10px;
            flex-wrap: wrap;
            margin-bottom: 20px;
        }
        .veiculo-option {
            flex: 1;
            min-width: 80px;
            text-align: center;
            padding: 15px 10px;
            border: 2px solid #e0e0e0;
            border-radius: 12px;
            cursor: pointer;
            transition: all 0.3s;
        }
        .veiculo-option:hover {
            border-color: #0d47a1;
        }
        .veiculo-option.selected {
            border-color: #0d47a1;
            background: #e3f2fd;
        }
        .veiculo-option .icon {
            font-size: 30px;
        }
        .veiculo-option .label {
            font-size: 12px;
            margin-top: 5px;
        }
        .veiculo-option input {
            display: none;
        }
        .dados-veiculo {
            display: none;
        }
        .dados-veiculo.show {
            display: block;
        }
        .sucesso-box {
            text-align: center;
            padding: 40px 20px;
        }
        .sucesso-box .icon {
            font-size: 80px;
            margin-bottom: 20px;
        }
        .sucesso-box h3 {
            color: #0d47a1;
            margin-bottom: 15px;
        }
        .info-box {
            background: #fff3e0;
            border-left: 4px solid #ff9800;
            padding: 15px;
            border-radius: 8px;
            margin: 20px 0;
        }
        @media (max-width: 500px) {
            .form-row { flex-direction: column; gap: 0; }
            .veiculo-option { min-width: 60px; padding: 10px 5px; }
            .veiculo-option .icon { font-size: 24px; }
        }
    </style>
</head>
<body>
    <div class="cadastro-container">
        <div class="cadastro-box">
            <?php if ($sucesso): ?>
                <div class="sucesso-box">
                    <div class="icon">📧</div>
                    <h3>Cadastro Enviado!</h3>
                    <p>Recebemos seu cadastro com sucesso.</p>
                    
                    <div class="info-box">
                        <strong>⏳ Aguarde a aprovação</strong><br>
                        Seu cadastro será analisado em até <strong>72 horas</strong>.<br>
                        Você receberá um email quando for aprovado.
                    </div>
                    
                    <p style="color: #666; font-size: 14px;">
                        Enviamos um email de confirmação para<br>
                        <strong><?= htmlspecialchars($email) ?></strong>
                    </p>
                    
                    <a href="index.php" class="btn" style="background: #0d47a1; color: white; margin-top: 20px;">
                        Voltar ao Login
                    </a>
                </div>
            <?php else: ?>
                <div style="text-align: center; font-size: 50px; margin-bottom: 10px;">🏍️</div>
                <h2>Cadastro de Entregador</h2>
                <p class="subtitle">Preencha seus dados para se tornar um entregador</p>
                
                <?php if ($erro): ?>
                    <div class="alert alert-danger"><?= htmlspecialchars($erro) ?></div>
                <?php endif; ?>
                
                <form method="POST" enctype="multipart/form-data">
                    <!-- Dados Pessoais -->
                    <h4 class="section-title">👤 Dados Pessoais</h4>
                    
                    <div class="form-group">
                        <label>Nome Completo *</label>
                        <input type="text" name="nome" class="form-control" required value="<?= htmlspecialchars($_POST['nome'] ?? '') ?>">
                    </div>
                    
                    <div class="form-row">
                        <div class="form-group">
                            <label>Email *</label>
                            <input type="email" name="email" class="form-control" required value="<?= htmlspecialchars($_POST['email'] ?? '') ?>">
                        </div>
                        <div class="form-group">
                            <label>Telefone *</label>
                            <input type="tel" name="telefone" class="form-control" required placeholder="(11) 99999-9999" value="<?= htmlspecialchars($_POST['telefone'] ?? '') ?>">
                        </div>
                    </div>
                    
                    <div class="form-group">
                        <label>CPF *</label>
                        <input type="text" name="cpf" class="form-control" required placeholder="000.000.000-00" value="<?= htmlspecialchars($_POST['cpf'] ?? '') ?>">
                    </div>
                    
                    <div class="form-group">
                        <label>Endereço Completo *</label>
                        <input type="text" name="endereco" class="form-control" required placeholder="Rua, número, bairro, cidade - UF" value="<?= htmlspecialchars($_POST['endereco'] ?? '') ?>">
                    </div>
                    
                    <div class="form-row">
                        <div class="form-group">
                            <label>Senha *</label>
                            <input type="password" name="senha" class="form-control" required minlength="6">
                        </div>
                        <div class="form-group">
                            <label>Confirmar Senha *</label>
                            <input type="password" name="confirmar" class="form-control" required>
                        </div>
                    </div>
                    
                    <!-- Tipo de Veículo -->
                    <h4 class="section-title">🚗 Tipo de Veículo</h4>
                    
                    <div class="veiculo-options">
                        <label class="veiculo-option" data-veiculo="bicicleta">
                            <input type="radio" name="veiculo" value="bicicleta">
                            <div class="icon">🚲</div>
                            <div class="label">Bicicleta</div>
                        </label>
                        <label class="veiculo-option selected" data-veiculo="moto">
                            <input type="radio" name="veiculo" value="moto" checked>
                            <div class="icon">🏍️</div>
                            <div class="label">Moto</div>
                        </label>
                        <label class="veiculo-option" data-veiculo="carro">
                            <input type="radio" name="veiculo" value="carro">
                            <div class="icon">🚗</div>
                            <div class="label">Carro</div>
                        </label>
                        <label class="veiculo-option" data-veiculo="van">
                            <input type="radio" name="veiculo" value="van">
                            <div class="icon">🚐</div>
                            <div class="label">Van</div>
                        </label>
                        <label class="veiculo-option" data-veiculo="caminhao">
                            <input type="radio" name="veiculo" value="caminhao">
                            <div class="icon">🚚</div>
                            <div class="label">Caminhão</div>
                        </label>
                    </div>
                    
                    <!-- Dados do Veículo (não aparece para bicicleta) -->
                    <div class="dados-veiculo show" id="dados-veiculo">
                        <h4 class="section-title">📋 Dados do Veículo</h4>
                        
                        <div class="form-row">
                            <div class="form-group">
                                <label>Placa *</label>
                                <input type="text" name="placa" id="placa" class="form-control" placeholder="ABC-1234 ou ABC1D23" value="<?= htmlspecialchars($_POST['placa'] ?? '') ?>">
                                <small style="color: #666;">Aceita formato antigo e Mercosul</small>
                            </div>
                            <div class="form-group">
                                <label>Ano *</label>
                                <input type="text" name="ano" class="form-control" placeholder="2020" maxlength="4" value="<?= htmlspecialchars($_POST['ano'] ?? '') ?>">
                            </div>
                        </div>
                        
                        <div class="form-row">
                            <div class="form-group">
                                <label>Marca *</label>
                                <input type="text" name="marca" class="form-control" placeholder="Honda, Fiat..." value="<?= htmlspecialchars($_POST['marca'] ?? '') ?>">
                            </div>
                            <div class="form-group">
                                <label>Modelo *</label>
                                <input type="text" name="modelo" class="form-control" placeholder="CG 160, Uno..." value="<?= htmlspecialchars($_POST['modelo'] ?? '') ?>">
                            </div>
                        </div>
                        
                        <div class="form-group">
                            <label>Cor *</label>
                            <input type="text" name="cor" class="form-control" placeholder="Preto, Branco, Vermelho..." value="<?= htmlspecialchars($_POST['cor'] ?? '') ?>">
                        </div>
                    </div>
                    
                    <!-- Documentos -->
                    <h4 class="section-title">📄 Documentos</h4>
                    <p style="color: #666; font-size: 13px; margin-bottom: 15px;">
                        Envie fotos ou PDF dos documentos. Máximo 5MB cada.
                    </p>
                    
                    <div class="form-row">
                        <div class="form-group">
                            <label class="file-input" id="fi-identidade">
                                <input type="file" name="doc_identidade" accept=".jpg,.jpeg,.png,.pdf" required>
                                <div class="icon">🪪</div>
                                <div class="text">RG / Identidade *</div>
                            </label>
                        </div>
                        <div class="form-group">
                            <label class="file-input" id="fi-cpf">
                                <input type="file" name="doc_cpf" accept=".jpg,.jpeg,.png,.pdf" required>
                                <div class="icon">📄</div>
                                <div class="text">CPF *</div>
                            </label>
                        </div>
                    </div>
                    
                    <div class="form-row docs-veiculo" id="docs-veiculo">
                        <div class="form-group">
                            <label class="file-input" id="fi-cnh">
                                <input type="file" name="doc_cnh" accept=".jpg,.jpeg,.png,.pdf">
                                <div class="icon">🚗</div>
                                <div class="text">CNH *</div>
                            </label>
                        </div>
                        <div class="form-group">
                            <label class="file-input" id="fi-veiculo">
                                <input type="file" name="doc_veiculo" accept=".jpg,.jpeg,.png,.pdf">
                                <div class="icon">📋</div>
                                <div class="text">CRLV do Veículo *</div>
                            </label>
                        </div>
                    </div>
                    
                    <div class="form-group">
                        <label class="file-input" id="fi-endereco">
                            <input type="file" name="doc_endereco" accept=".jpg,.jpeg,.png,.pdf">
                            <div class="icon">🏠</div>
                            <div class="text">Comprovante de Endereço (opcional)</div>
                        </label>
                    </div>
                    
                    <div class="info-box" style="margin-top: 20px;">
                        <strong>⏳ Prazo de aprovação</strong><br>
                        Após o cadastro, seus documentos serão analisados em até <strong>72 horas</strong>.
                        Você receberá um email quando seu cadastro for aprovado.
                    </div>
                    
                    <button type="submit" class="btn btn-block" style="background: #0d47a1; color: white; margin-top: 20px;">
                        Enviar Cadastro
                    </button>
                </form>
                
                <p style="text-align: center; margin-top: 20px; color: #666;">
                    Já tem conta? <a href="index.php" style="color: #0d47a1;">Faça login</a>
                </p>
            <?php endif; ?>
        </div>
    </div>
    
    <script>
    // Seleção de tipo de veículo
    document.querySelectorAll('.veiculo-option').forEach(option => {
        option.addEventListener('click', function() {
            document.querySelectorAll('.veiculo-option').forEach(o => o.classList.remove('selected'));
            this.classList.add('selected');
            
            const veiculo = this.dataset.veiculo;
            const dadosVeiculo = document.getElementById('dados-veiculo');
            const docsVeiculo = document.getElementById('docs-veiculo');
            
            if (veiculo === 'bicicleta') {
                dadosVeiculo.classList.remove('show');
                docsVeiculo.style.display = 'none';
                // Remover required dos campos de veículo
                document.querySelector('[name="placa"]').removeAttribute('required');
                document.querySelector('[name="doc_cnh"]').removeAttribute('required');
                document.querySelector('[name="doc_veiculo"]').removeAttribute('required');
            } else {
                dadosVeiculo.classList.add('show');
                docsVeiculo.style.display = 'flex';
                document.querySelector('[name="placa"]').setAttribute('required', '');
                document.querySelector('[name="doc_cnh"]').setAttribute('required', '');
                document.querySelector('[name="doc_veiculo"]').setAttribute('required', '');
            }
        });
    });
    
    // Feedback visual de arquivo selecionado
    document.querySelectorAll('.file-input input').forEach(input => {
        input.addEventListener('change', function() {
            const label = this.closest('.file-input');
            if (this.files.length > 0) {
                label.classList.add('selected');
                label.querySelector('.text').textContent = this.files[0].name;
            } else {
                label.classList.remove('selected');
            }
        });
    });
    
    // Máscara de placa
    document.getElementById('placa').addEventListener('input', function() {
        let v = this.value.toUpperCase().replace(/[^A-Z0-9]/g, '');
        if (v.length > 7) v = v.substr(0, 7);
        this.value = v;
    });
    </script>
</body>
</html>
