<?php
require_once '../config/database.php';
require_once '../config/auth.php';
requireLogin('entregador', 'index.php');

$db = Database::getInstance()->getConnection();
$entregadorId = getCurrentUserId('entregador');

// Verificar se entregador está bloqueado
$stmt = $db->prepare("SELECT bloqueado, motivo_bloqueio FROM entregadores WHERE id = ?");
$stmt->execute([$entregadorId]);
$entregadorStatus = $stmt->fetch(PDO::FETCH_ASSOC);

if (!empty($entregadorStatus['bloqueado']) && $entregadorStatus['bloqueado'] == 1) {
    // Fazer logout e redirecionar
    session_destroy();
    session_start();
    $_SESSION['erro_bloqueio'] = '🚫 Atenção! Seu acesso à plataforma foi bloqueado temporariamente devido a uma verificação necessária em sua conta. Para entender o motivo e solicitar a liberação, entre em contato com o suporte. Nossa equipe está disponível para orientar você.';
    header('Location: index.php');
    exit;
}

// Buscar status atual
$stmt = $db->prepare("SELECT status, latitude, longitude FROM entregadores WHERE id = ?");
$stmt->execute([$entregadorId]);
$entregador = $stmt->fetch(PDO::FETCH_ASSOC);
$statusAtual = $entregador['status'];

// Buscar entrega em andamento
$stmt = $db->prepare("
    SELECT e.*, u.nome as cliente_nome, u.telefone as cliente_telefone
    FROM entregas e
    JOIN usuarios u ON e.usuario_id = u.id
    WHERE e.entregador_id = ? AND e.status IN ('aceita', 'coletada')
    LIMIT 1
");
$stmt->execute([$entregadorId]);
$entregaAtual = $stmt->fetch(PDO::FETCH_ASSOC);

// Buscar entregas disponíveis (se não tiver entrega em andamento)
$entregasDisponiveis = [];
if (!$entregaAtual && $statusAtual === 'disponivel') {
    $stmt = $db->prepare("
        SELECT e.*, u.nome as cliente_nome
        FROM entregas e
        JOIN usuarios u ON e.usuario_id = u.id
        WHERE e.status = 'pendente'
        ORDER BY e.created_at DESC
    ");
    $stmt->execute();
    $entregasDisponiveis = $stmt->fetchAll(PDO::FETCH_ASSOC);
}

// Ganhos do dia (líquido)
$stmt = $db->prepare("
    SELECT COALESCE(SUM(valor_liquido), 0) as liquido, COALESCE(SUM(valor_bruto), 0) as bruto
    FROM ganhos_entregadores
    WHERE entregador_id = ? AND DATE(data_pagamento) = CURDATE()
");
$stmt->execute([$entregadorId]);
$ganhos = $stmt->fetch(PDO::FETCH_ASSOC);
$ganhosHoje = $ganhos['liquido'];
$ganhosBrutoHoje = $ganhos['bruto'];

// Buscar configurações de preço
$configs = [];
$stmt = $db->query("SELECT chave, valor FROM configuracoes");
while ($row = $stmt->fetch(PDO::FETCH_ASSOC)) {
    $configs[$row['chave']] = $row['valor'];
}

// Preços por categoria
$precosPorKm = [
    'moto' => floatval($configs['preco_km_moto'] ?? 2.00),
    'carro' => floatval($configs['preco_km_carro'] ?? 2.50),
    'van' => floatval($configs['preco_km_van'] ?? 3.50),
    'caminhao' => floatval($configs['preco_km_caminhao'] ?? 5.00)
];

// API Key do Google Maps
$googleMapsKey = 'AIzaSyBQyClNByvKjHubTBizQ1GquLjhwSyHz2k';
?>
<!DOCTYPE html>
<html lang="pt-BR">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Dashboard - Já EntreGo Entregador</title>
    
    <!-- PWA Meta Tags -->
    <meta name="theme-color" content="#0d47a1">
    <meta name="apple-mobile-web-app-capable" content="yes">
    <meta name="apple-mobile-web-app-status-bar-style" content="black-translucent">
    <meta name="apple-mobile-web-app-title" content="EntreGo Driver">
    <link rel="manifest" href="../manifest-entregador.json">
    <link rel="apple-touch-icon" href="../assets/icons/icon-192.png">
    
    <link rel="stylesheet" href="../assets/css/style.css">
    <link rel="stylesheet" href="../assets/css/entregador.css">
    <style>
        #mapa {
            width: 100%;
            height: 250px;
            border-radius: 12px;
            margin-bottom: 16px;
        }
        .nav-buttons {
            display: flex;
            gap: 10px;
            margin-bottom: 16px;
        }
        .nav-btn {
            flex: 1;
            display: flex;
            align-items: center;
            justify-content: center;
            gap: 8px;
            padding: 14px;
            border: none;
            border-radius: 10px;
            font-size: 14px;
            font-weight: 600;
            cursor: pointer;
            text-decoration: none;
            color: white;
        }
        .nav-btn.waze {
            background: linear-gradient(135deg, #33ccff 0%, #00aaff 100%);
        }
        .nav-btn.gmaps {
            background: linear-gradient(135deg, #4285f4 0%, #34a853 100%);
        }
        .nav-btn img {
            width: 24px;
            height: 24px;
        }
        .destino-atual {
            background: linear-gradient(135deg, #0d47a1 0%, #1565c0 100%);
            color: white;
            padding: 12px 16px;
            border-radius: 10px;
            margin-bottom: 12px;
            display: flex;
            align-items: center;
            gap: 10px;
        }
        .destino-atual .icone {
            font-size: 24px;
        }
        .destino-atual .info {
            flex: 1;
        }
        .destino-atual .label {
            font-size: 11px;
            opacity: 0.8;
            text-transform: uppercase;
        }
        .destino-atual .endereco {
            font-size: 14px;
            font-weight: 500;
        }
        /* Botão de Status */
        .btn-status {
            padding: 10px 20px;
            border: none;
            border-radius: 25px;
            font-size: 14px;
            font-weight: 600;
            cursor: pointer;
            transition: all 0.3s;
        }
        .btn-status.online {
            background: linear-gradient(135deg, #00c853 0%, #00a844 100%);
            color: white;
        }
        .btn-status.offline {
            background: linear-gradient(135deg, #ff5252 0%, #ff1744 100%);
            color: white;
        }
        .btn-status:hover {
            transform: scale(1.05);
            box-shadow: 0 4px 15px rgba(0,0,0,0.2);
        }
        
        /* Botões de ação da entrega */
        .acoes-entrega-container {
            display: flex;
            flex-direction: column;
            gap: 10px;
        }
        
        .btn-cancelar-entrega {
            background: linear-gradient(135deg, #f44336 0%, #d32f2f 100%);
            color: white;
            border: none;
            padding: 12px 20px;
            border-radius: 10px;
            font-size: 14px;
            font-weight: 600;
            cursor: pointer;
            transition: all 0.3s;
        }
        
        .btn-cancelar-entrega:hover {
            transform: translateY(-2px);
            box-shadow: 0 4px 15px rgba(244, 67, 54, 0.3);
        }
        
        /* Modal de cancelamento */
        .modal-overlay {
            position: fixed;
            top: 0;
            left: 0;
            right: 0;
            bottom: 0;
            background: rgba(0, 0, 0, 0.7);
            display: none;
            align-items: center;
            justify-content: center;
            z-index: 10000;
            padding: 20px;
        }
        
        .modal-overlay.show {
            display: flex;
        }
        
        .modal-content {
            background: white;
            border-radius: 16px;
            width: 100%;
            max-width: 400px;
            max-height: 90vh;
            overflow-y: auto;
            animation: modalSlideIn 0.3s ease;
        }
        
        @keyframes modalSlideIn {
            from {
                opacity: 0;
                transform: scale(0.9) translateY(20px);
            }
            to {
                opacity: 1;
                transform: scale(1) translateY(0);
            }
        }
        
        .modal-header {
            display: flex;
            justify-content: space-between;
            align-items: center;
            padding: 20px 20px 0;
            border-bottom: 1px solid #e0e0e0;
            margin-bottom: 20px;
        }
        
        .modal-header h3 {
            color: #f44336;
            margin: 0;
            font-size: 18px;
        }
        
        .modal-close {
            background: none;
            border: none;
            font-size: 24px;
            color: #666;
            cursor: pointer;
            padding: 0;
            width: 30px;
            height: 30px;
            display: flex;
            align-items: center;
            justify-content: center;
        }
        
        .modal-body {
            padding: 0 20px 20px;
        }
        
        .motivos-cancelamento {
            display: flex;
            flex-direction: column;
            gap: 8px;
        }
        
        .motivo-item {
            display: flex;
            align-items: center;
            gap: 12px;
            padding: 12px;
            border: 2px solid #e0e0e0;
            border-radius: 10px;
            cursor: pointer;
            transition: all 0.3s;
        }
        
        .motivo-item:hover {
            border-color: #f44336;
            background: #fff5f5;
        }
        
        .motivo-item input[type="radio"] {
            display: none;
        }
        
        .motivo-item input[type="radio"]:checked + .motivo-icon + .motivo-texto {
            color: #f44336;
            font-weight: 600;
        }
        
        .motivo-item input[type="radio"]:checked ~ * {
            color: #f44336;
        }
        
        .motivo-item:has(input[type="radio"]:checked) {
            border-color: #f44336;
            background: #fff5f5;
        }
        
        .motivo-icon {
            font-size: 20px;
            width: 24px;
            text-align: center;
        }
        
        .motivo-texto {
            flex: 1;
            font-size: 14px;
            color: #333;
        }
        
        .motivo-personalizado {
            margin-top: 15px;
        }
        
        .motivo-personalizado textarea {
            width: 100%;
            min-height: 80px;
            padding: 12px;
            border: 2px solid #e0e0e0;
            border-radius: 8px;
            font-family: inherit;
            font-size: 14px;
            resize: vertical;
        }
        
        .motivo-personalizado textarea:focus {
            outline: none;
            border-color: #f44336;
        }
        
        .contador-chars {
            text-align: right;
            font-size: 12px;
            color: #666;
            margin-top: 5px;
        }
        
        .modal-footer {
            display: flex;
            gap: 10px;
            padding: 20px;
            border-top: 1px solid #e0e0e0;
        }
        
        .btn-modal-cancelar {
            flex: 1;
            background: #f5f5f5;
            color: #666;
            border: none;
            padding: 12px;
            border-radius: 8px;
            font-size: 14px;
            font-weight: 600;
            cursor: pointer;
        }
        
        .btn-modal-confirmar {
            flex: 1;
            background: linear-gradient(135deg, #f44336 0%, #d32f2f 100%);
            color: white;
            border: none;
            padding: 12px;
            border-radius: 8px;
            font-size: 14px;
            font-weight: 600;
            cursor: pointer;
        }
    </style>
</head>
<body class="entregador">
    <!-- Header -->
    <header class="header-entregador">
        <h1>
            <span class="status-indicator" style="background: <?= $statusAtual === 'disponivel' ? '#00c853' : '#ff1744' ?>"></span>
            Já EntreGo
        </h1>
        <span><?= htmlspecialchars(getCurrentUserName('entregador')) ?></span>
    </header>
    
    <!-- Notificação Popup -->
    <div class="notificacao-popup" id="notificacao">
        <div class="icon">🔔</div>
        <div class="texto">
            <div class="titulo">Nova Entrega!</div>
            <div class="subtitulo" id="notificacao-texto"></div>
        </div>
    </div>
    
    <main class="main-content">
        <!-- Toggle Status -->
        <div class="status-toggle">
            <span>Status: <strong id="status-texto"><?= $statusAtual === 'disponivel' ? 'Online' : 'Offline' ?></strong></span>
            <button type="button" id="btn-toggle-status" class="btn-status <?= $statusAtual === 'disponivel' ? 'online' : 'offline' ?>">
                <?= $statusAtual === 'disponivel' ? '🟢 Online' : '🔴 Offline' ?>
            </button>
        </div>
        
        <?php if ($entregaAtual): ?>
            <!-- Entrega em andamento com Mapa -->
            <div class="entrega-andamento" style="overflow: visible;">
                <div class="header-status">
                    <h3>Entrega em Andamento</h3>
                    <div class="status"><?= $entregaAtual['status'] === 'aceita' ? '📦 Ir para Coleta' : '🚚 Ir para Entrega' ?></div>
                </div>
                <div class="conteudo">
                    <!-- Destino Atual -->
                    <div class="destino-atual">
                        <div class="icone"><?= $entregaAtual['status'] === 'aceita' ? '📦' : '🏠' ?></div>
                        <div class="info">
                            <div class="label"><?= $entregaAtual['status'] === 'aceita' ? 'Ir para coleta' : 'Ir para entrega' ?></div>
                            <div class="endereco">
                                <?= htmlspecialchars($entregaAtual['status'] === 'aceita' ? $entregaAtual['origem_endereco'] : $entregaAtual['destino_endereco']) ?>
                            </div>
                        </div>
                    </div>
                    
                    <!-- Mapa -->
                    <div id="mapa"></div>
                    
                    <!-- Botões de Navegação -->
                    <div class="nav-buttons">
                        <a href="#" id="btn-waze" class="nav-btn waze" target="_blank">
                            🗺️ Waze
                        </a>
                        <a href="#" id="btn-gmaps" class="nav-btn gmaps" target="_blank">
                            📍 Google Maps
                        </a>
                    </div>
                    
                    <!-- Rota completa -->
                    <div class="rota">
                        <div class="ponto">
                            <div class="ponto-icon origem">A</div>
                            <div class="ponto-info">
                                <div class="ponto-label">Coleta</div>
                                <div class="ponto-endereco"><?= htmlspecialchars($entregaAtual['origem_endereco']) ?></div>
                            </div>
                        </div>
                        <div class="linha-conexao"></div>
                        <div class="ponto">
                            <div class="ponto-icon destino">B</div>
                            <div class="ponto-info">
                                <div class="ponto-label">Entrega</div>
                                <div class="ponto-endereco"><?= htmlspecialchars($entregaAtual['destino_endereco']) ?></div>
                            </div>
                        </div>
                    </div>
                    
                    <div style="background: #e3f2fd; padding: 12px; border-radius: 8px; margin: 16px 0;">
                        <strong>Cliente:</strong> <?= htmlspecialchars($entregaAtual['cliente_nome']) ?>
                        <?php if ($entregaAtual['cliente_telefone']): ?>
                            <br><strong>Tel:</strong> 
                            <a href="tel:<?= htmlspecialchars($entregaAtual['cliente_telefone']) ?>" style="color: #0d47a1;">
                                <?= htmlspecialchars($entregaAtual['cliente_telefone']) ?>
                            </a>
                        <?php endif; ?>
                    </div>
                    
                    <!-- Informações da Entrega -->
                    <div style="background: #f8f9fa; padding: 16px; border-radius: 12px; margin: 16px 0;">
                        <div style="text-align: center; margin-bottom: 12px;">
                            <div style="font-size: 28px; font-weight: bold; color: #00c853;">
                                R$ <?= number_format($entregaAtual['valor'], 2, ',', '.') ?>
                            </div>
                            <div style="font-size: 12px; color: #666; margin-top: 4px;">Valor Total</div>
                        </div>
                        
                        <?php if (!empty($entregaAtual['distancia'])): ?>
                        <div style="display: flex; justify-content: space-between; align-items: center; padding: 8px 0; border-top: 1px solid #e0e0e0;">
                            <span style="color: #666; font-size: 14px;">📏 Distância:</span>
                            <span style="font-weight: 600; color: #333;"><?= number_format($entregaAtual['distancia'], 1, ',', '.') ?> km</span>
                        </div>
                        <?php endif; ?>
                        
                        <?php 
                        $categoria = $entregaAtual['categoria_veiculo'] ?? 'moto';
                        $valorPorKm = $precosPorKm[$categoria] ?? 2.00;
                        ?>
                        <div style="display: flex; justify-content: space-between; align-items: center; padding: 8px 0; border-top: 1px solid #e0e0e0;">
                            <span style="color: #666; font-size: 14px;">💰 Valor/km:</span>
                            <span style="font-weight: 600; color: #1a73e8;">R$ <?= number_format($valorPorKm, 2, ',', '.') ?></span>
                        </div>
                        
                        <div style="display: flex; justify-content: space-between; align-items: center; padding: 8px 0; border-top: 1px solid #e0e0e0;">
                            <span style="color: #666; font-size: 14px;">🚗 Categoria:</span>
                            <span style="font-weight: 600; color: #333; text-transform: capitalize;"><?= $categoria ?></span>
                        </div>
                    </div>
                    
                    <div class="acoes-entrega-container">
                        <?php if ($entregaAtual['status'] === 'aceita'): ?>
                            <button class="btn-acao-entrega btn-coletar" onclick="atualizarStatus(<?= $entregaAtual['id'] ?>, 'coletada')">
                                ✓ Confirmar Coleta
                            </button>
                            <button class="btn-cancelar-entrega" onclick="abrirModalCancelamento(<?= $entregaAtual['id'] ?>)">
                                ✕ Cancelar Entrega
                            </button>
                        <?php else: ?>
                            <button class="btn-acao-entrega btn-entregar" onclick="atualizarStatus(<?= $entregaAtual['id'] ?>, 'entregue')">
                                ✓ Confirmar Entrega
                            </button>
                        <?php endif; ?>
                    </div>
                </div>
            </div>
            
            <script>
            // Dados da entrega para o mapa
            const entregaAtual = {
                status: '<?= $entregaAtual['status'] ?>',
                origem: '<?= addslashes($entregaAtual['origem_endereco']) ?>',
                destino: '<?= addslashes($entregaAtual['destino_endereco']) ?>',
                origemLat: <?= $entregaAtual['origem_lat'] ?: 'null' ?>,
                origemLng: <?= $entregaAtual['origem_lng'] ?: 'null' ?>,
                destinoLat: <?= $entregaAtual['destino_lat'] ?: 'null' ?>,
                destinoLng: <?= $entregaAtual['destino_lng'] ?: 'null' ?>
            };
            </script>
        <?php else: ?>
            <!-- Ganhos do dia (só mostra quando não tem entrega) -->
            <div class="ganhos-card">
                <div class="label">Ganhos Líquidos de Hoje</div>
                <div class="valor">R$ <?= number_format($ganhosHoje, 2, ',', '.') ?></div>
                <div class="periodo"><?= date('d/m/Y') ?></div>
                <?php if ($ganhosBrutoHoje > 0): ?>
                    <div style="font-size: 12px; opacity: 0.7; margin-top: 5px;">
                        Bruto: R$ <?= number_format($ganhosBrutoHoje, 2, ',', '.') ?>
                    </div>
                <?php endif; ?>
            </div>
            
            <?php if ($statusAtual === 'disponivel'): ?>
                <!-- Entregas disponíveis -->
                <div id="entregas-container">
                    <?php if (empty($entregasDisponiveis)): ?>
                        <div class="sem-entregas">
                            <div class="icon">📭</div>
                            <h3>Nenhuma entrega disponível</h3>
                            <p>Aguarde, novas entregas aparecerão aqui</p>
                        </div>
                    <?php else: ?>
                        <?php foreach ($entregasDisponiveis as $entrega): ?>
                            <div class="entrega-card nova" id="entrega-<?= $entrega['id'] ?>">
                                <!-- Valor e Informações -->
                                <div style="display: flex; justify-content: space-between; align-items: center; margin-bottom: 16px;">
                                    <div class="valor">R$ <?= number_format($entrega['valor'], 2, ',', '.') ?></div>
                                    <?php if (!empty($entrega['distancia'])): ?>
                                    <div style="text-align: right; font-size: 12px; color: #666;">
                                        <div>📏 <?= number_format($entrega['distancia'], 1, ',', '.') ?> km</div>
                                        <?php 
                                        $categoria = $entrega['categoria_veiculo'] ?? 'moto';
                                        $valorPorKm = $precosPorKm[$categoria] ?? 2.00;
                                        ?>
                                        <div>💰 R$ <?= number_format($valorPorKm, 2, ',', '.') ?>/km</div>
                                    </div>
                                    <?php endif; ?>
                                </div>
                                
                                <div class="rota">
                                    <div class="ponto">
                                        <div class="ponto-icon origem">A</div>
                                        <div class="ponto-info">
                                            <div class="ponto-label">Coleta</div>
                                            <div class="ponto-endereco"><?= htmlspecialchars($entrega['origem_endereco']) ?></div>
                                        </div>
                                    </div>
                                    <div class="linha-conexao"></div>
                                    <div class="ponto">
                                        <div class="ponto-icon destino">B</div>
                                        <div class="ponto-info">
                                            <div class="ponto-label">Entrega</div>
                                            <div class="ponto-endereco"><?= htmlspecialchars($entrega['destino_endereco']) ?></div>
                                        </div>
                                    </div>
                                </div>
                                
                                <?php if ($entrega['descricao']): ?>
                                    <div style="color: #666; font-size: 14px; margin-bottom: 16px;">
                                        📦 <?= htmlspecialchars($entrega['descricao']) ?>
                                    </div>
                                <?php endif; ?>
                                
                                <div class="acoes-entrega">
                                    <button class="btn-aceitar" onclick="aceitarEntrega(<?= $entrega['id'] ?>)">Aceitar</button>
                                </div>
                            </div>
                        <?php endforeach; ?>
                    <?php endif; ?>
                </div>
            <?php else: ?>
                <div class="sem-entregas">
                    <div class="icon">😴</div>
                    <h3>Você está offline</h3>
                    <p>Ative o status para receber entregas</p>
                </div>
            <?php endif; ?>
            
            <script>const entregaAtual = null;</script>
        <?php endif; ?>
    </main>
    
    <!-- Modal de Cancelamento -->
    <div class="modal-overlay" id="modal-cancelamento">
        <div class="modal-content">
            <div class="modal-header">
                <h3>⚠️ Cancelar Entrega</h3>
                <button class="modal-close" onclick="fecharModalCancelamento()">✕</button>
            </div>
            <div class="modal-body">
                <p style="color: #666; margin-bottom: 20px;">
                    Selecione o motivo do cancelamento. Esta ação não pode ser desfeita.
                </p>
                
                <div class="motivos-cancelamento">
                    <label class="motivo-item">
                        <input type="radio" name="motivo" value="Rota muito longa - Distância maior que o esperado">
                        <span class="motivo-icon">🛣️</span>
                        <span class="motivo-texto">Rota muito longa</span>
                    </label>
                    
                    <label class="motivo-item">
                        <input type="radio" name="motivo" value="Área de risco - Local perigoso para entrega">
                        <span class="motivo-icon">⚠️</span>
                        <span class="motivo-texto">Área de risco</span>
                    </label>
                    
                    <label class="motivo-item">
                        <input type="radio" name="motivo" value="Problema no veículo - Pneu furou">
                        <span class="motivo-icon">🔧</span>
                        <span class="motivo-texto">Pneu furou</span>
                    </label>
                    
                    <label class="motivo-item">
                        <input type="radio" name="motivo" value="Problema no veículo - Falha mecânica">
                        <span class="motivo-icon">⚙️</span>
                        <span class="motivo-texto">Problema mecânico</span>
                    </label>
                    
                    <label class="motivo-item">
                        <input type="radio" name="motivo" value="Combustível insuficiente - Preciso abastecer">
                        <span class="motivo-icon">⛽</span>
                        <span class="motivo-texto">Sem combustível</span>
                    </label>
                    
                    <label class="motivo-item">
                        <input type="radio" name="motivo" value="Emergência pessoal - Situação urgente">
                        <span class="motivo-icon">🚨</span>
                        <span class="motivo-texto">Emergência pessoal</span>
                    </label>
                    
                    <label class="motivo-item">
                        <input type="radio" name="motivo" value="Condições climáticas - Chuva forte/temporal">
                        <span class="motivo-icon">🌧️</span>
                        <span class="motivo-texto">Chuva forte</span>
                    </label>
                    
                    <label class="motivo-item">
                        <input type="radio" name="motivo" value="Trânsito intenso - Congestionamento severo">
                        <span class="motivo-icon">🚦</span>
                        <span class="motivo-texto">Trânsito intenso</span>
                    </label>
                    
                    <label class="motivo-item">
                        <input type="radio" name="motivo" value="Outro motivo">
                        <span class="motivo-icon">📝</span>
                        <span class="motivo-texto">Outro motivo</span>
                    </label>
                </div>
                
                <div class="motivo-personalizado" id="motivo-personalizado" style="display: none;">
                    <textarea placeholder="Descreva o motivo do cancelamento..." maxlength="200"></textarea>
                    <div class="contador-chars">0/200</div>
                </div>
            </div>
            <div class="modal-footer">
                <button class="btn-modal-cancelar" onclick="fecharModalCancelamento()">
                    Voltar
                </button>
                <button class="btn-modal-confirmar" onclick="confirmarCancelamento()">
                    Cancelar Entrega
                </button>
            </div>
        </div>
    </div>

    <!-- Bottom Navigation -->
    <nav class="bottom-nav">
        <a href="dashboard.php" class="active">
            <span class="icon">🏠</span>
            Início
        </a>
        <a href="historico.php">
            <span class="icon">📋</span>
            Histórico
        </a>
        <a href="ganhos.php">
            <span class="icon">💰</span>
            Ganhos
        </a>
        <a href="logout.php">
            <span class="icon">🚪</span>
            Sair
        </a>
    </nav>
    
    <!-- Google Maps API -->
    <script src="https://maps.googleapis.com/maps/api/js?key=<?= $googleMapsKey ?>&libraries=places,geometry&callback=initMap" async defer></script>
    
    <script>
    let map, directionsService, directionsRenderer;
    let marcadorEntregador, posicaoAtual;
    
    // Inicializar mapa
    function initMap() {
        if (!entregaAtual) return;
        
        directionsService = new google.maps.DirectionsService();
        directionsRenderer = new google.maps.DirectionsRenderer({
            suppressMarkers: false,
            polylineOptions: {
                strokeColor: '#0d47a1',
                strokeWeight: 5
            }
        });
        
        // Posição padrão (será atualizada com geolocalização)
        const defaultPos = { lat: -23.5505, lng: -46.6333 };
        
        map = new google.maps.Map(document.getElementById('mapa'), {
            zoom: 14,
            center: defaultPos,
            disableDefaultUI: true,
            zoomControl: true,
            styles: [
                { featureType: "poi", stylers: [{ visibility: "off" }] }
            ]
        });
        
        directionsRenderer.setMap(map);
        
        // Obter localização atual
        if (navigator.geolocation) {
            navigator.geolocation.getCurrentPosition(
                (position) => {
                    posicaoAtual = {
                        lat: position.coords.latitude,
                        lng: position.coords.longitude
                    };
                    atualizarRota();
                    atualizarLinksNavegacao();
                    
                    // Atualizar posição no servidor
                    fetch('api/atualizar-localizacao.php', {
                        method: 'POST',
                        headers: {'Content-Type': 'application/x-www-form-urlencoded'},
                        body: `lat=${posicaoAtual.lat}&lng=${posicaoAtual.lng}`
                    });
                },
                () => {
                    // Sem permissão, usar geocoding do endereço
                    geocodeEndereco();
                },
                { enableHighAccuracy: true }
            );
            
            // Atualizar posição continuamente
            navigator.geolocation.watchPosition((position) => {
                posicaoAtual = {
                    lat: position.coords.latitude,
                    lng: position.coords.longitude
                };
                atualizarLinksNavegacao();
            }, null, { enableHighAccuracy: true });
        } else {
            geocodeEndereco();
        }
    }
    
    // Geocodificar endereço se não tiver coordenadas
    function geocodeEndereco() {
        const geocoder = new google.maps.Geocoder();
        const endereco = entregaAtual.status === 'aceita' ? entregaAtual.origem : entregaAtual.destino;
        
        geocoder.geocode({ address: endereco + ', Brasil' }, (results, status) => {
            if (status === 'OK' && results[0]) {
                const loc = results[0].geometry.location;
                map.setCenter(loc);
                
                new google.maps.Marker({
                    position: loc,
                    map: map,
                    title: endereco
                });
            }
        });
    }
    
    // Atualizar rota no mapa
    function atualizarRota() {
        if (!posicaoAtual) return;
        
        const destino = entregaAtual.status === 'aceita' ? entregaAtual.origem : entregaAtual.destino;
        
        directionsService.route({
            origin: posicaoAtual,
            destination: destino + ', Brasil',
            travelMode: google.maps.TravelMode.DRIVING
        }, (result, status) => {
            if (status === 'OK') {
                directionsRenderer.setDirections(result);
            }
        });
    }
    
    // Atualizar links de navegação
    function atualizarLinksNavegacao() {
        const destino = entregaAtual.status === 'aceita' ? entregaAtual.origem : entregaAtual.destino;
        const destinoEncoded = encodeURIComponent(destino);
        
        // Link Waze
        let wazeUrl;
        if (posicaoAtual) {
            wazeUrl = `https://waze.com/ul?q=${destinoEncoded}&navigate=yes`;
        } else {
            wazeUrl = `https://waze.com/ul?q=${destinoEncoded}`;
        }
        document.getElementById('btn-waze').href = wazeUrl;
        
        // Link Google Maps
        let gmapsUrl;
        if (posicaoAtual) {
            gmapsUrl = `https://www.google.com/maps/dir/?api=1&origin=${posicaoAtual.lat},${posicaoAtual.lng}&destination=${destinoEncoded}&travelmode=driving`;
        } else {
            gmapsUrl = `https://www.google.com/maps/search/?api=1&query=${destinoEncoded}`;
        }
        document.getElementById('btn-gmaps').href = gmapsUrl;
    }
    
    // Toggle status online/offline
    let statusOnline = <?= $statusAtual === 'disponivel' ? 'true' : 'false' ?>;
    
    document.getElementById('btn-toggle-status').addEventListener('click', function() {
        const btn = this;
        const novoStatus = statusOnline ? 'offline' : 'disponivel';
        
        btn.disabled = true;
        btn.textContent = '⏳ Aguarde...';
        
        fetch('api/atualizar-status.php', {
            method: 'POST',
            headers: {'Content-Type': 'application/x-www-form-urlencoded'},
            body: 'status=' + novoStatus
        })
        .then(r => r.json())
        .then(data => {
            if (data.success) {
                statusOnline = !statusOnline;
                document.getElementById('status-texto').textContent = statusOnline ? 'Online' : 'Offline';
                btn.textContent = statusOnline ? '🟢 Online' : '🔴 Offline';
                btn.className = 'btn-status ' + (statusOnline ? 'online' : 'offline');
                
                // Mostrar mensagem
                alert(statusOnline ? '✅ Você está online!' : '👋 Você desconectou!');
                
                // Recarregar para mostrar entregas
                setTimeout(() => location.reload(), 300);
            } else {
                alert('Erro ao atualizar status');
                btn.textContent = statusOnline ? '🟢 Online' : '🔴 Offline';
            }
            btn.disabled = false;
        })
        .catch(() => {
            alert('Erro de conexão');
            btn.textContent = statusOnline ? '🟢 Online' : '🔴 Offline';
            btn.disabled = false;
        });
    });
    
    // Aceitar entrega
    function aceitarEntrega(id) {
        fetch('api/aceitar-entrega.php', {
            method: 'POST',
            headers: {'Content-Type': 'application/x-www-form-urlencoded'},
            body: 'entrega_id=' + id
        }).then(r => r.json()).then(data => {
            if (data.success) {
                location.reload();
            } else {
                alert(data.message || 'Erro ao aceitar entrega');
            }
        });
    }
    
    // Atualizar status da entrega
    function atualizarStatus(id, status) {
        fetch('api/atualizar-entrega.php', {
            method: 'POST',
            headers: {'Content-Type': 'application/x-www-form-urlencoded'},
            body: 'entrega_id=' + id + '&status=' + status
        }).then(r => r.json()).then(data => {
            if (data.success) {
                // Se foi entrega concluída, mostrar informações de ganho
                if (status === 'entregue' && data.ganho) {
                    const ganho = data.ganho;
                    const valorBruto = ganho.valor_bruto.toFixed(2).replace('.', ',');
                    const valorTaxa = ganho.valor_taxa.toFixed(2).replace('.', ',');
                    const valorLiquido = ganho.valor_liquido.toFixed(2).replace('.', ',');
                    const taxaPercent = ganho.taxa_percentual.toFixed(1);
                    
                    alert(`🎉 Entrega concluída com sucesso!\n\n💰 Resumo do seu ganho:\n\n💵 Valor bruto: R$ ${valorBruto}\n📉 Taxa da plataforma (${taxaPercent}%): R$ ${valorTaxa}\n✅ Valor líquido recebido: R$ ${valorLiquido}\n\nO valor já foi creditado em sua conta!`);
                }
                location.reload();
            } else {
                alert(data.message || 'Erro ao atualizar');
            }
        });
    }
    
    // Verificar novas entregas periodicamente
    function verificarNovasEntregas() {
        fetch('api/verificar-notificacoes.php')
            .then(r => r.json())
            .then(data => {
                console.log('🔔 Verificando notificações:', data); // Debug
                if (data.novas && data.novas.length > 0) {
                    console.log('🎉 Novas entregas encontradas:', data.novas.length); // Debug
                    data.novas.forEach(notif => {
                        console.log('📦 Processando notificação:', notif.mensagem); // Debug
                        mostrarNotificacao(notif.mensagem);
                        tocarSomComFallback();
                        
                        // Notificação do navegador
                        if ('Notification' in window && Notification.permission === 'granted') {
                            new Notification('Nova Entrega!', {
                                body: notif.mensagem,
                                icon: '../assets/icons/icon-192.png',
                                tag: 'nova-entrega-' + notif.id,
                                requireInteraction: true
                            });
                        }
                    });
                    setTimeout(() => location.reload(), 3000);
                }
            })
            .catch(err => {
                console.error('❌ Erro ao verificar notificações:', err); // Debug
            });
    }
    
    // Mostrar notificação visual
    function mostrarNotificacao(texto) {
        const notif = document.getElementById('notificacao');
        document.getElementById('notificacao-texto').textContent = texto;
        notif.classList.add('show');
        setTimeout(() => notif.classList.remove('show'), 5000);
    }
    
    // Sistema de som melhorado com múltiplos fallbacks
    let audioContext = null;
    let audioInitialized = false;
    
    // Inicializar contexto de áudio na primeira interação
    function initAudio() {
        if (!audioInitialized) {
            try {
                audioContext = new (window.AudioContext || window.webkitAudioContext)();
                audioInitialized = true;
                console.log('🔊 Contexto de áudio inicializado');
            } catch (e) {
                console.log('❌ Erro ao inicializar contexto de áudio:', e);
            }
        }
    }
    
    // Tocar som com múltiplos fallbacks
    function tocarSomComFallback() {
        console.log('🔊 Tentando tocar som de notificação...'); // Debug
        
        // Vibração primeiro (sempre funciona)
        if ('vibrate' in navigator) {
            navigator.vibrate([300, 100, 300, 100, 500]);
            console.log('📳 Vibração ativada');
        }
        
        // Tentar som personalizado primeiro
        tentarSomPersonalizado()
            .then(success => {
                if (!success) {
                    console.log('🎵 Fallback para som sintético');
                    tocarSomSintetico();
                }
            })
            .catch(() => {
                console.log('🎵 Fallback para som sintético (catch)');
                tocarSomSintetico();
            });
    }
    
    // Tentar tocar som personalizado
    function tentarSomPersonalizado() {
        return new Promise((resolve) => {
            try {
                // Testar múltiplos caminhos
                const caminhos = [
                    '../assets/sounds/notificacao.mp3',
                    '/assets/sounds/notificacao.mp3',
                    'assets/sounds/notificacao.mp3'
                ];
                
                let tentativa = 0;
                
                function tentarProximoCaminho() {
                    if (tentativa >= caminhos.length) {
                        console.log('❌ Todos os caminhos de som falharam');
                        resolve(false);
                        return;
                    }
                    
                    const audio = new Audio(caminhos[tentativa]);
                    audio.volume = 0.9;
                    audio.preload = 'auto';
                    
                    audio.addEventListener('canplaythrough', () => {
                        console.log('✅ Som carregado:', caminhos[tentativa]);
                        audio.play()
                            .then(() => {
                                console.log('🎵 Som personalizado tocado com sucesso!');
                                resolve(true);
                            })
                            .catch(err => {
                                console.log('❌ Erro ao reproduzir:', err);
                                tentativa++;
                                tentarProximoCaminho();
                            });
                    });
                    
                    audio.addEventListener('error', (e) => {
                        console.log('❌ Erro ao carregar som:', caminhos[tentativa], e);
                        tentativa++;
                        tentarProximoCaminho();
                    });
                    
                    // Timeout de 2 segundos por tentativa
                    setTimeout(() => {
                        if (audio.readyState < 4) {
                            console.log('⏰ Timeout no carregamento:', caminhos[tentativa]);
                            tentativa++;
                            tentarProximoCaminho();
                        }
                    }, 2000);
                }
                
                tentarProximoCaminho();
                
            } catch (e) {
                console.log('❌ Erro geral ao criar áudio:', e);
                resolve(false);
            }
        });
    }
    
    // Som sintético melhorado como fallback
    function tocarSomSintetico() {
        try {
            console.log('🎵 Tocando som sintético...');
            
            // Usar contexto existente ou criar novo
            const ctx = audioContext || new (window.AudioContext || window.webkitAudioContext)();
            
            // Sequência de notas mais chamativa
            const sequencia = [
                { freq: 880, duracao: 0.2, delay: 0 },      // A5
                { freq: 1100, duracao: 0.2, delay: 0.15 },  // C#6
                { freq: 880, duracao: 0.3, delay: 0.3 },    // A5 (mais longo)
                { freq: 1320, duracao: 0.4, delay: 0.5 }    // E6 (final)
            ];
            
            sequencia.forEach((nota, index) => {
                setTimeout(() => {
                    try {
                        const osc = ctx.createOscillator();
                        const gain = ctx.createGain();
                        const filter = ctx.createBiquadFilter();
                        
                        // Conectar: oscilador -> filtro -> ganho -> saída
                        osc.connect(filter);
                        filter.connect(gain);
                        gain.connect(ctx.destination);
                        
                        // Configurar oscilador
                        osc.frequency.setValueAtTime(nota.freq, ctx.currentTime);
                        osc.type = index === sequencia.length - 1 ? 'sawtooth' : 'square';
                        
                        // Configurar filtro (suavizar o som)
                        filter.type = 'lowpass';
                        filter.frequency.setValueAtTime(2000, ctx.currentTime);
                        
                        // Envelope de volume (fade in/out)
                        gain.gain.setValueAtTime(0, ctx.currentTime);
                        gain.gain.linearRampToValueAtTime(0.7, ctx.currentTime + 0.05);
                        gain.gain.linearRampToValueAtTime(0.5, ctx.currentTime + nota.duracao * 0.7);
                        gain.gain.linearRampToValueAtTime(0, ctx.currentTime + nota.duracao);
                        
                        osc.start(ctx.currentTime);
                        osc.stop(ctx.currentTime + nota.duracao);
                        
                        console.log(`🎵 Nota ${index + 1}: ${nota.freq}Hz`);
                    } catch (e) {
                        console.error(`❌ Erro na nota ${index + 1}:`, e);
                    }
                }, nota.delay * 1000);
            });
            
            console.log('✅ Som sintético iniciado com sucesso');
        } catch(e) {
            console.error('❌ Erro crítico no som sintético:', e);
            // Último recurso: beep do sistema
            try {
                console.log('🔔 Tentando beep do sistema...');
                const audio = new Audio('data:audio/wav;base64,UklGRnoGAABXQVZFZm10IBAAAAABAAEAQB8AAEAfAAABAAgAZGF0YQoGAACBhYqFbF1fdJivrJBhNjVgodDbq2EcBj+a2/LDciUFLIHO8tiJNwgZaLvt559NEAxQp+PwtmMcBjiR1/LMeSwFJHfH8N2QQAoUXrTp66hVFApGn+DyvmwhBSuBzvLZiTYIG2m98OScTgwOUarm7blmGgU7k9n1unEiBC13yO/eizEIHWq+8+OWT');
                audio.play().catch(() => console.log('❌ Beep do sistema também falhou'));
            } catch (beepError) {
                console.log('❌ Beep do sistema não disponível');
            }
        }
    }
    
    // Inicializar áudio na primeira interação do usuário
    document.addEventListener('click', initAudio, { once: true });
    document.addEventListener('touchstart', initAudio, { once: true });
    
    // Event listeners para o modal de cancelamento
    document.addEventListener('DOMContentLoaded', function() {
        // Mostrar campo personalizado quando "Outro motivo" for selecionado
        document.querySelectorAll('input[name="motivo"]').forEach(radio => {
            radio.addEventListener('change', function() {
                const motivoPersonalizado = document.getElementById('motivo-personalizado');
                if (this.value === 'Outro motivo') {
                    motivoPersonalizado.style.display = 'block';
                    motivoPersonalizado.querySelector('textarea').focus();
                } else {
                    motivoPersonalizado.style.display = 'none';
                }
            });
        });
        
        // Contador de caracteres para textarea
        const textarea = document.querySelector('#motivo-personalizado textarea');
        const contador = document.querySelector('.contador-chars');
        
        textarea.addEventListener('input', function() {
            const length = this.value.length;
            contador.textContent = `${length}/200`;
            
            if (length > 180) {
                contador.style.color = '#f44336';
            } else {
                contador.style.color = '#666';
            }
        });
        
        // Fechar modal ao clicar fora
        document.getElementById('modal-cancelamento').addEventListener('click', function(e) {
            if (e.target === this) {
                fecharModalCancelamento();
            }
        });
    });
    
    // Verificar a cada 3 segundos (mais frequente)
    setInterval(verificarNovasEntregas, 3000);
    
    // Solicitar permissões
    if ('Notification' in window && Notification.permission === 'default') {
        Notification.requestPermission().then(permission => {
            console.log('🔔 Permissão de notificação:', permission);
        });
    }
    
    // Função de teste de som
    function testarSom() {
        console.log('🧪 TESTE MANUAL DE SOM INICIADO');
        initAudio();
        mostrarNotificacao('Teste de som e notificação');
        tocarSomComFallback();
    }
    
    // Variável global para armazenar ID da entrega sendo cancelada
    let entregaCancelando = null;
    
    // Abrir modal de cancelamento
    function abrirModalCancelamento(entregaId) {
        entregaCancelando = entregaId;
        document.getElementById('modal-cancelamento').classList.add('show');
        document.body.style.overflow = 'hidden';
    }
    
    // Fechar modal de cancelamento
    function fecharModalCancelamento() {
        document.getElementById('modal-cancelamento').classList.remove('show');
        document.body.style.overflow = '';
        entregaCancelando = null;
        
        // Limpar seleções
        document.querySelectorAll('input[name="motivo"]').forEach(radio => {
            radio.checked = false;
        });
        document.getElementById('motivo-personalizado').style.display = 'none';
        document.querySelector('#motivo-personalizado textarea').value = '';
    }
    
    // Confirmar cancelamento
    function confirmarCancelamento() {
        const motivoSelecionado = document.querySelector('input[name="motivo"]:checked');
        
        if (!motivoSelecionado) {
            alert('Por favor, selecione um motivo para o cancelamento');
            return;
        }
        
        let motivo = motivoSelecionado.value;
        
        // Se for "Outro motivo", pegar o texto personalizado
        if (motivo === 'Outro motivo') {
            const textoPersonalizado = document.querySelector('#motivo-personalizado textarea').value.trim();
            if (!textoPersonalizado) {
                alert('Por favor, descreva o motivo do cancelamento');
                return;
            }
            motivo = textoPersonalizado;
        }
        
        // Confirmar ação
        if (!confirm('Tem certeza que deseja cancelar esta entrega?\n\nMotivo: ' + motivo)) {
            return;
        }
        
        // Desabilitar botão para evitar cliques duplos
        const btnConfirmar = document.querySelector('.btn-modal-confirmar');
        btnConfirmar.disabled = true;
        btnConfirmar.textContent = 'Cancelando...';
        
        // Enviar cancelamento
        fetch('api/cancelar-entrega.php', {
            method: 'POST',
            headers: {'Content-Type': 'application/x-www-form-urlencoded'},
            body: `entrega_id=${entregaCancelando}&motivo=${encodeURIComponent(motivo)}`
        })
        .then(r => r.json())
        .then(data => {
            if (data.success) {
                alert('✅ Entrega cancelada com sucesso!\n\nO cliente foi notificado sobre o cancelamento.');
                fecharModalCancelamento();
                location.reload();
            } else {
                alert('❌ Erro: ' + (data.message || 'Não foi possível cancelar a entrega'));
                btnConfirmar.disabled = false;
                btnConfirmar.textContent = 'Cancelar Entrega';
            }
        })
        .catch(err => {
            console.error('Erro ao cancelar:', err);
            alert('❌ Erro de conexão. Tente novamente.');
            btnConfirmar.disabled = false;
            btnConfirmar.textContent = 'Cancelar Entrega';
        });
    }
    
    // Teste de som no carregamento (após interação)
    setTimeout(() => {
        if (audioInitialized) {
            console.log('🧪 Sistema de áudio pronto para uso');
        }
    }, 2000);
    </script>
    
    <!-- PWA Script -->
    <script src="../assets/js/pwa.js"></script>
</body>
</html>
