<?php
require_once '../config/database.php';
require_once '../config/auth.php';
require_once '../config/helpers.php';
requireLogin('entregador', 'index.php');

$db = Database::getInstance()->getConnection();
$entregadorId = getCurrentUserId('entregador');

// Buscar taxa atual
$taxaPercentual = floatval(getConfig('taxa_entregador_percentual', 15));
$taxaFixa = floatval(getConfig('taxa_entregador_fixa', 0));

// Ganhos de hoje (usando tabela de ganhos)
$stmt = $db->prepare("
    SELECT 
        COALESCE(SUM(g.valor_liquido), 0) as liquido,
        COALESCE(SUM(g.valor_bruto), 0) as bruto,
        COALESCE(SUM(g.valor_taxa), 0) as taxa,
        COUNT(*) as qtd
    FROM ganhos_entregadores g
    WHERE g.entregador_id = ? AND DATE(g.data_pagamento) = CURDATE()
");
$stmt->execute([$entregadorId]);
$hoje = $stmt->fetch(PDO::FETCH_ASSOC);

// Ganhos da semana
$stmt = $db->prepare("
    SELECT 
        COALESCE(SUM(g.valor_liquido), 0) as liquido,
        COALESCE(SUM(g.valor_bruto), 0) as bruto,
        COALESCE(SUM(g.valor_taxa), 0) as taxa,
        COUNT(*) as qtd
    FROM ganhos_entregadores g
    WHERE g.entregador_id = ? AND YEARWEEK(g.data_pagamento) = YEARWEEK(CURDATE())
");
$stmt->execute([$entregadorId]);
$semana = $stmt->fetch(PDO::FETCH_ASSOC);

// Ganhos do mês
$stmt = $db->prepare("
    SELECT 
        COALESCE(SUM(g.valor_liquido), 0) as liquido,
        COALESCE(SUM(g.valor_bruto), 0) as bruto,
        COALESCE(SUM(g.valor_taxa), 0) as taxa,
        COUNT(*) as qtd
    FROM ganhos_entregadores g
    WHERE g.entregador_id = ? 
    AND MONTH(g.data_pagamento) = MONTH(CURDATE()) 
    AND YEAR(g.data_pagamento) = YEAR(CURDATE())
");
$stmt->execute([$entregadorId]);
$mes = $stmt->fetch(PDO::FETCH_ASSOC);

// Ganhos por dia (últimos 7 dias)
$stmt = $db->prepare("
    SELECT 
        DATE(g.data_pagamento) as dia, 
        SUM(g.valor_liquido) as liquido,
        SUM(g.valor_bruto) as bruto,
        COUNT(*) as qtd
    FROM ganhos_entregadores g
    WHERE g.entregador_id = ? 
    AND g.data_pagamento >= DATE_SUB(CURDATE(), INTERVAL 7 DAY)
    GROUP BY DATE(g.data_pagamento)
    ORDER BY dia DESC
");
$stmt->execute([$entregadorId]);
$porDia = $stmt->fetchAll(PDO::FETCH_ASSOC);

// Ganhos detalhados (últimas 10 entregas)
$stmt = $db->prepare("
    SELECT 
        g.*,
        e.origem_endereco,
        e.destino_endereco,
        e.updated_at as data_entrega
    FROM ganhos_entregadores g
    JOIN entregas e ON g.entrega_id = e.id
    WHERE g.entregador_id = ?
    ORDER BY g.data_pagamento DESC
    LIMIT 10
");
$stmt->execute([$entregadorId]);
$ganhosDetalhados = $stmt->fetchAll(PDO::FETCH_ASSOC);
?>
<!DOCTYPE html>
<html lang="pt-BR">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Ganhos - Entregador</title>
    <link rel="stylesheet" href="../assets/css/style.css">
    <link rel="stylesheet" href="../assets/css/entregador.css">
</head>
<body class="entregador">
    <header class="header-entregador">
        <h1>💰 Meus Ganhos</h1>
    </header>
    
    <main class="main-content">
        <!-- Info Taxa -->
        <div style="background: #fff3e0; margin: 16px; padding: 12px 16px; border-radius: 8px; font-size: 13px; color: #e65100;">
            💡 Taxa da plataforma: <?= $taxaPercentual ?>%<?= $taxaFixa > 0 ? ' + R$ ' . number_format($taxaFixa, 2, ',', '.') : '' ?> por entrega
        </div>
        
        <!-- Ganhos Hoje -->
        <div class="ganhos-card">
            <div class="label">Ganho Líquido Hoje</div>
            <div class="valor">R$ <?= number_format($hoje['liquido'], 2, ',', '.') ?></div>
            <div class="periodo"><?= $hoje['qtd'] ?> entregas</div>
            <div style="font-size: 12px; opacity: 0.7; margin-top: 5px;">
                Bruto: R$ <?= number_format($hoje['bruto'], 2, ',', '.') ?> | 
                Taxa: R$ <?= number_format($hoje['taxa'], 2, ',', '.') ?>
            </div>
        </div>
        
        <!-- Resumo -->
        <div style="display: flex; gap: 16px; margin: 16px;">
            <div style="flex: 1; background: white; border-radius: 12px; padding: 16px; text-align: center;">
                <div style="color: #666; font-size: 12px;">Esta Semana</div>
                <div style="font-size: 20px; font-weight: bold; color: #0d47a1;">R$ <?= number_format($semana['liquido'], 2, ',', '.') ?></div>
                <div style="font-size: 12px; color: #999;"><?= $semana['qtd'] ?> entregas</div>
            </div>
            <div style="flex: 1; background: white; border-radius: 12px; padding: 16px; text-align: center;">
                <div style="color: #666; font-size: 12px;">Este Mês</div>
                <div style="font-size: 20px; font-weight: bold; color: #0d47a1;">R$ <?= number_format($mes['liquido'], 2, ',', '.') ?></div>
                <div style="font-size: 12px; color: #999;"><?= $mes['qtd'] ?> entregas</div>
            </div>
        </div>
        
        <!-- Últimos 7 dias -->
        <div style="background: white; margin: 16px; border-radius: 12px; padding: 16px;">
            <h3 style="margin-bottom: 16px; color: #333;">Últimos 7 dias</h3>
            <?php if (empty($porDia)): ?>
                <p style="color: #666; text-align: center;">Nenhum ganho registrado</p>
            <?php else: ?>
                <?php foreach ($porDia as $dia): ?>
                    <div style="display: flex; justify-content: space-between; align-items: center; padding: 12px 0; border-bottom: 1px solid #eee;">
                        <div>
                            <div style="color: #333; font-weight: 500;"><?= date('d/m', strtotime($dia['dia'])) ?></div>
                            <div style="color: #666; font-size: 12px;"><?= $dia['qtd'] ?> entregas</div>
                        </div>
                        <div style="text-align: right;">
                            <div style="font-weight: bold; color: #00c853;">R$ <?= number_format($dia['liquido'], 2, ',', '.') ?></div>
                            <div style="font-size: 12px; color: #999;">Bruto: R$ <?= number_format($dia['bruto'], 2, ',', '.') ?></div>
                        </div>
                    </div>
                <?php endforeach; ?>
            <?php endif; ?>
        </div>
        
        <!-- Ganhos Detalhados -->
        <div style="background: white; margin: 16px; border-radius: 12px; padding: 16px;">
            <h3 style="margin-bottom: 16px; color: #333;">Últimas Entregas</h3>
            <?php if (empty($ganhosDetalhados)): ?>
                <p style="color: #666; text-align: center;">Nenhuma entrega concluída ainda</p>
            <?php else: ?>
                <?php foreach ($ganhosDetalhados as $ganho): ?>
                    <div style="border: 1px solid #e0e0e0; border-radius: 8px; padding: 12px; margin-bottom: 12px;">
                        <div style="display: flex; justify-content: space-between; align-items: flex-start; margin-bottom: 8px;">
                            <div style="flex: 1;">
                                <div style="font-size: 12px; color: #666; margin-bottom: 4px;">
                                    <?= date('d/m/Y H:i', strtotime($ganho['data_entrega'])) ?>
                                </div>
                                <div style="font-size: 13px; color: #333;">
                                    📍 <?= htmlspecialchars(substr($ganho['origem_endereco'], 0, 30)) ?>...
                                </div>
                                <div style="font-size: 13px; color: #333;">
                                    🏠 <?= htmlspecialchars(substr($ganho['destino_endereco'], 0, 30)) ?>...
                                </div>
                            </div>
                            <div style="text-align: right;">
                                <div style="font-weight: bold; color: #00c853; font-size: 16px;">
                                    R$ <?= number_format($ganho['valor_liquido'], 2, ',', '.') ?>
                                </div>
                                <div style="font-size: 11px; color: #666;">
                                    Bruto: R$ <?= number_format($ganho['valor_bruto'], 2, ',', '.') ?>
                                </div>
                                <div style="font-size: 11px; color: #f44336;">
                                    Taxa: R$ <?= number_format($ganho['valor_taxa'], 2, ',', '.') ?> (<?= number_format($ganho['taxa_percentual'], 1) ?>%)
                                </div>
                            </div>
                        </div>
                    </div>
                <?php endforeach; ?>
            <?php endif; ?>
        </div>
    </main>
    
    <nav class="bottom-nav">
        <a href="dashboard.php">
            <span class="icon">🏠</span>
            Início
        </a>
        <a href="historico.php">
            <span class="icon">📋</span>
            Histórico
        </a>
        <a href="ganhos.php" class="active">
            <span class="icon">💰</span>
            Ganhos
        </a>
        <a href="logout.php">
            <span class="icon">🚪</span>
            Sair
        </a>
    </nav>
</body>
</html>
