import React, { useState, useEffect } from 'react';
import {
  View,
  Text,
  StyleSheet,
  SafeAreaView,
  ScrollView,
  TouchableOpacity,
  TextInput,
  Alert,
  ActivityIndicator,
  Modal,
} from 'react-native';
import * as Location from 'expo-location';
import { colors, fonts, spacing, borderRadius, shadows } from '../../config/theme';
import { clienteAPI } from '../../config/api';

const CATEGORIAS_VEICULO = [
  { id: 'moto', nome: 'Moto', icon: '🏍️', desc: 'Até 15kg', preco: 'A partir de R$ 8' },
  { id: 'carro', nome: 'Carro', icon: '🚗', desc: 'Até 50kg', preco: 'A partir de R$ 12' },
  { id: 'van', nome: 'Van', icon: '🚐', desc: 'Até 500kg', preco: 'A partir de R$ 20' },
  { id: 'caminhao', nome: 'Caminhão', icon: '🚛', desc: 'Até 3000kg', preco: 'A partir de R$ 35' },
];

const TIPOS_CARGA = [
  { id: 'documentos', nome: 'Documentos', icon: '📄' },
  { id: 'eletronicos', nome: 'Eletrônicos', icon: '📱' },
  { id: 'roupas', nome: 'Roupas', icon: '👕' },
  { id: 'alimentos', nome: 'Alimentos', icon: '🍔' },
  { id: 'moveis', nome: 'Móveis', icon: '🛋️' },
  { id: 'fragil', nome: 'Frágil', icon: '⚠️' },
  { id: 'outros', nome: 'Outros', icon: '📦' },
];

export default function NovaEntregaScreen({ navigation }) {
  const [origem, setOrigem] = useState('');
  const [destino, setDestino] = useState('');
  const [descricao, setDescricao] = useState('');
  const [categoriaSelecionada, setCategoriaSelecionada] = useState('moto');
  const [tipoCarga, setTipoCarga] = useState('outros');
  const [peso, setPeso] = useState('');
  const [loading, setLoading] = useState(false);
  const [loadingLocation, setLoadingLocation] = useState(false);
  const [showTipoCargaModal, setShowTipoCargaModal] = useState(false);

  const usarMinhaLocalizacao = async () => {
    setLoadingLocation(true);
    try {
      const { status } = await Location.requestForegroundPermissionsAsync();
      if (status !== 'granted') {
        Alert.alert('Erro', 'Permissão de localização negada');
        return;
      }

      const location = await Location.getCurrentPositionAsync({});
      const [address] = await Location.reverseGeocodeAsync({
        latitude: location.coords.latitude,
        longitude: location.coords.longitude,
      });

      if (address) {
        const endereco = `${address.street || ''}, ${address.streetNumber || ''} - ${address.district || ''}, ${address.city || ''}`;
        setOrigem(endereco.replace(/^, /, '').replace(/, ,/g, ','));
      }
    } catch (error) {
      Alert.alert('Erro', 'Não foi possível obter sua localização');
      console.error(error);
    } finally {
      setLoadingLocation(false);
    }
  };

  const handleSolicitar = async () => {
    if (!origem || !destino) {
      Alert.alert('Erro', 'Preencha os endereços de origem e destino');
      return;
    }

    setLoading(true);
    try {
      const dados = `origem=${encodeURIComponent(origem)}&destino=${encodeURIComponent(destino)}&descricao=${encodeURIComponent(descricao)}&categoria=${categoriaSelecionada}&tipo_carga=${tipoCarga}&peso=${peso}`;
      
      const response = await clienteAPI.criarEntrega(dados);
      
      if (response.data.success) {
        Alert.alert(
          'Sucesso! 🎉',
          `Entrega solicitada!\nValor: R$ ${response.data.valor}\nVeículo: ${response.data.veiculo}`,
          [{ text: 'OK', onPress: () => navigation.navigate('Dashboard') }]
        );
      } else {
        Alert.alert('Erro', response.data.message || 'Erro ao solicitar entrega');
      }
    } catch (error) {
      Alert.alert('Erro', 'Não foi possível conectar ao servidor');
      console.error(error);
    } finally {
      setLoading(false);
    }
  };

  return (
    <SafeAreaView style={styles.container}>
      {/* Header */}
      <View style={styles.header}>
        <Text style={styles.headerTitle}>Nova Entrega</Text>
        <Text style={styles.headerSubtitle}>Preencha os dados da sua entrega</Text>
      </View>

      <ScrollView style={styles.content} showsVerticalScrollIndicator={false}>
        {/* Endereço de Origem */}
        <View style={styles.section}>
          <Text style={styles.sectionTitle}>📍 Endereço de Coleta</Text>
          <View style={styles.inputContainer}>
            <TextInput
              style={styles.input}
              placeholder="Digite o endereço de coleta"
              placeholderTextColor={colors.gray[400]}
              value={origem}
              onChangeText={setOrigem}
              multiline
            />
          </View>
          <TouchableOpacity
            style={styles.locationBtn}
            onPress={usarMinhaLocalizacao}
            disabled={loadingLocation}
          >
            {loadingLocation ? (
              <ActivityIndicator size="small" color={colors.primary} />
            ) : (
              <>
                <Text style={styles.locationIcon}>📍</Text>
                <Text style={styles.locationText}>Usar minha localização</Text>
              </>
            )}
          </TouchableOpacity>
        </View>

        {/* Endereço de Destino */}
        <View style={styles.section}>
          <Text style={styles.sectionTitle}>🏠 Endereço de Entrega</Text>
          <View style={styles.inputContainer}>
            <TextInput
              style={styles.input}
              placeholder="Digite o endereço de entrega"
              placeholderTextColor={colors.gray[400]}
              value={destino}
              onChangeText={setDestino}
              multiline
            />
          </View>
        </View>

        {/* Tipo de Carga */}
        <View style={styles.section}>
          <Text style={styles.sectionTitle}>📦 O que você vai enviar?</Text>
          <TouchableOpacity
            style={styles.selectBtn}
            onPress={() => setShowTipoCargaModal(true)}
          >
            <Text style={styles.selectIcon}>
              {TIPOS_CARGA.find(t => t.id === tipoCarga)?.icon}
            </Text>
            <Text style={styles.selectText}>
              {TIPOS_CARGA.find(t => t.id === tipoCarga)?.nome}
            </Text>
            <Text style={styles.selectArrow}>▼</Text>
          </TouchableOpacity>
        </View>

        {/* Peso Aproximado */}
        <View style={styles.section}>
          <Text style={styles.sectionTitle}>⚖️ Peso aproximado (kg)</Text>
          <View style={styles.inputContainer}>
            <TextInput
              style={styles.input}
              placeholder="Ex: 5"
              placeholderTextColor={colors.gray[400]}
              value={peso}
              onChangeText={setPeso}
              keyboardType="numeric"
            />
          </View>
        </View>

        {/* Categoria de Veículo */}
        <View style={styles.section}>
          <Text style={styles.sectionTitle}>🚗 Tipo de Veículo</Text>
          <View style={styles.categoriasGrid}>
            {CATEGORIAS_VEICULO.map((cat) => (
              <TouchableOpacity
                key={cat.id}
                style={[
                  styles.categoriaCard,
                  categoriaSelecionada === cat.id && styles.categoriaCardSelected,
                ]}
                onPress={() => setCategoriaSelecionada(cat.id)}
              >
                <Text style={styles.categoriaIcon}>{cat.icon}</Text>
                <Text style={[
                  styles.categoriaNome,
                  categoriaSelecionada === cat.id && styles.categoriaTextSelected,
                ]}>
                  {cat.nome}
                </Text>
                <Text style={styles.categoriaDesc}>{cat.desc}</Text>
                <Text style={[
                  styles.categoriaPreco,
                  categoriaSelecionada === cat.id && styles.categoriaTextSelected,
                ]}>
                  {cat.preco}
                </Text>
              </TouchableOpacity>
            ))}
          </View>
        </View>

        {/* Descrição */}
        <View style={styles.section}>
          <Text style={styles.sectionTitle}>📝 Descrição (opcional)</Text>
          <View style={styles.inputContainer}>
            <TextInput
              style={[styles.input, styles.textArea]}
              placeholder="Detalhes sobre a entrega..."
              placeholderTextColor={colors.gray[400]}
              value={descricao}
              onChangeText={setDescricao}
              multiline
              numberOfLines={3}
            />
          </View>
        </View>

        {/* Botão Solicitar */}
        <TouchableOpacity
          style={[styles.submitBtn, loading && styles.submitBtnDisabled]}
          onPress={handleSolicitar}
          disabled={loading}
        >
          {loading ? (
            <ActivityIndicator color={colors.white} />
          ) : (
            <Text style={styles.submitBtnText}>Solicitar Entrega</Text>
          )}
        </TouchableOpacity>

        <View style={{ height: 40 }} />
      </ScrollView>

      {/* Modal Tipo de Carga */}
      <Modal
        visible={showTipoCargaModal}
        transparent
        animationType="slide"
        onRequestClose={() => setShowTipoCargaModal(false)}
      >
        <View style={styles.modalOverlay}>
          <View style={styles.modalContent}>
            <Text style={styles.modalTitle}>Selecione o tipo de carga</Text>
            {TIPOS_CARGA.map((tipo) => (
              <TouchableOpacity
                key={tipo.id}
                style={[
                  styles.modalItem,
                  tipoCarga === tipo.id && styles.modalItemSelected,
                ]}
                onPress={() => {
                  setTipoCarga(tipo.id);
                  setShowTipoCargaModal(false);
                }}
              >
                <Text style={styles.modalItemIcon}>{tipo.icon}</Text>
                <Text style={styles.modalItemText}>{tipo.nome}</Text>
                {tipoCarga === tipo.id && <Text style={styles.modalItemCheck}>✓</Text>}
              </TouchableOpacity>
            ))}
            <TouchableOpacity
              style={styles.modalCloseBtn}
              onPress={() => setShowTipoCargaModal(false)}
            >
              <Text style={styles.modalCloseBtnText}>Fechar</Text>
            </TouchableOpacity>
          </View>
        </View>
      </Modal>
    </SafeAreaView>
  );
}

const styles = StyleSheet.create({
  container: {
    flex: 1,
    backgroundColor: colors.background,
  },
  header: {
    backgroundColor: colors.primary,
    paddingHorizontal: spacing.lg,
    paddingVertical: spacing.lg,
    paddingTop: spacing.xl,
  },
  headerTitle: {
    fontSize: fonts.sizes.xxl,
    fontWeight: 'bold',
    color: colors.white,
  },
  headerSubtitle: {
    fontSize: fonts.sizes.sm,
    color: 'rgba(255, 255, 255, 0.8)',
    marginTop: 2,
  },
  content: {
    flex: 1,
    padding: spacing.lg,
  },
  section: {
    marginBottom: spacing.xl,
  },
  sectionTitle: {
    fontSize: fonts.sizes.md,
    fontWeight: '600',
    color: colors.gray[800],
    marginBottom: spacing.sm,
  },
  inputContainer: {
    backgroundColor: colors.white,
    borderRadius: borderRadius.lg,
    ...shadows.sm,
  },
  input: {
    padding: spacing.md,
    fontSize: fonts.sizes.md,
    color: colors.gray[900],
  },
  textArea: {
    minHeight: 80,
    textAlignVertical: 'top',
  },
  locationBtn: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'center',
    backgroundColor: colors.white,
    borderRadius: borderRadius.lg,
    padding: spacing.md,
    marginTop: spacing.sm,
    borderWidth: 1,
    borderColor: colors.primary,
    borderStyle: 'dashed',
  },
  locationIcon: {
    fontSize: 18,
    marginRight: spacing.sm,
  },
  locationText: {
    color: colors.primary,
    fontWeight: '600',
  },
  selectBtn: {
    flexDirection: 'row',
    alignItems: 'center',
    backgroundColor: colors.white,
    borderRadius: borderRadius.lg,
    padding: spacing.md,
    ...shadows.sm,
  },
  selectIcon: {
    fontSize: 24,
    marginRight: spacing.sm,
  },
  selectText: {
    flex: 1,
    fontSize: fonts.sizes.md,
    color: colors.gray[800],
  },
  selectArrow: {
    color: colors.gray[400],
  },
  categoriasGrid: {
    flexDirection: 'row',
    flexWrap: 'wrap',
    marginHorizontal: -spacing.xs,
  },
  categoriaCard: {
    width: '48%',
    backgroundColor: colors.white,
    borderRadius: borderRadius.lg,
    padding: spacing.md,
    margin: '1%',
    alignItems: 'center',
    borderWidth: 2,
    borderColor: 'transparent',
    ...shadows.sm,
  },
  categoriaCardSelected: {
    borderColor: colors.primary,
    backgroundColor: '#e3f2fd',
  },
  categoriaIcon: {
    fontSize: 32,
    marginBottom: spacing.xs,
  },
  categoriaNome: {
    fontSize: fonts.sizes.md,
    fontWeight: '600',
    color: colors.gray[800],
  },
  categoriaDesc: {
    fontSize: fonts.sizes.xs,
    color: colors.gray[500],
    marginTop: 2,
  },
  categoriaPreco: {
    fontSize: fonts.sizes.xs,
    color: colors.primary,
    fontWeight: '600',
    marginTop: spacing.xs,
  },
  categoriaTextSelected: {
    color: colors.primary,
  },
  submitBtn: {
    backgroundColor: colors.primary,
    borderRadius: borderRadius.lg,
    padding: spacing.lg,
    alignItems: 'center',
    ...shadows.md,
  },
  submitBtnDisabled: {
    opacity: 0.7,
  },
  submitBtnText: {
    color: colors.white,
    fontSize: fonts.sizes.lg,
    fontWeight: 'bold',
  },
  modalOverlay: {
    flex: 1,
    backgroundColor: 'rgba(0, 0, 0, 0.5)',
    justifyContent: 'flex-end',
  },
  modalContent: {
    backgroundColor: colors.white,
    borderTopLeftRadius: borderRadius.xl,
    borderTopRightRadius: borderRadius.xl,
    padding: spacing.xl,
    maxHeight: '70%',
  },
  modalTitle: {
    fontSize: fonts.sizes.lg,
    fontWeight: 'bold',
    color: colors.gray[900],
    marginBottom: spacing.lg,
    textAlign: 'center',
  },
  modalItem: {
    flexDirection: 'row',
    alignItems: 'center',
    padding: spacing.md,
    borderRadius: borderRadius.lg,
    marginBottom: spacing.sm,
  },
  modalItemSelected: {
    backgroundColor: '#e3f2fd',
  },
  modalItemIcon: {
    fontSize: 24,
    marginRight: spacing.md,
  },
  modalItemText: {
    flex: 1,
    fontSize: fonts.sizes.md,
    color: colors.gray[800],
  },
  modalItemCheck: {
    color: colors.primary,
    fontSize: fonts.sizes.lg,
    fontWeight: 'bold',
  },
  modalCloseBtn: {
    backgroundColor: colors.gray[200],
    borderRadius: borderRadius.lg,
    padding: spacing.md,
    alignItems: 'center',
    marginTop: spacing.md,
  },
  modalCloseBtnText: {
    color: colors.gray[700],
    fontWeight: '600',
  },
});
