import React, { useState, useEffect, useRef } from 'react';
import {
  View,
  Text,
  StyleSheet,
  SafeAreaView,
  ScrollView,
  TouchableOpacity,
  RefreshControl,
  Alert,
  Linking,
  Modal,
} from 'react-native';
import * as Location from 'expo-location';
import * as Haptics from 'expo-haptics';
import { Audio } from 'expo-av';
import AsyncStorage from '@react-native-async-storage/async-storage';
import { colors, fonts, spacing, borderRadius, shadows } from '../../config/theme';
import { entregadorAPI } from '../../config/api';

const MOTIVOS_CANCELAMENTO = [
  { id: 1, icon: '🛣️', texto: 'Rota muito longa', motivo: 'Rota muito longa - Distância maior que o esperado' },
  { id: 2, icon: '⚠️', texto: 'Área de risco', motivo: 'Área de risco - Local perigoso para entrega' },
  { id: 3, icon: '🔧', texto: 'Pneu furou', motivo: 'Problema no veículo - Pneu furou' },
  { id: 4, icon: '⚙️', texto: 'Problema mecânico', motivo: 'Problema no veículo - Falha mecânica' },
  { id: 5, icon: '⛽', texto: 'Sem combustível', motivo: 'Combustível insuficiente - Preciso abastecer' },
  { id: 6, icon: '🚨', texto: 'Emergência pessoal', motivo: 'Emergência pessoal - Situação urgente' },
  { id: 7, icon: '🌧️', texto: 'Chuva forte', motivo: 'Condições climáticas - Chuva forte/temporal' },
  { id: 8, icon: '🚦', texto: 'Trânsito intenso', motivo: 'Trânsito intenso - Congestionamento severo' },
];

export default function DashboardScreen() {
  const [entregador, setEntregador] = useState(null);
  const [entregaAtual, setEntregaAtual] = useState(null);
  const [entregasDisponiveis, setEntregasDisponiveis] = useState([]);
  const [ganhosHoje, setGanhosHoje] = useState(0);
  const [isOnline, setIsOnline] = useState(false);
  const [refreshing, setRefreshing] = useState(false);
  const [showCancelModal, setShowCancelModal] = useState(false);
  const [motivoSelecionado, setMotivoSelecionado] = useState(null);
  
  const notificationInterval = useRef(null);
  const soundRef = useRef(null);


  useEffect(() => {
    loadData();
    loadEntregadorData();
    startLocationTracking();
    
    return () => {
      if (notificationInterval.current) {
        clearInterval(notificationInterval.current);
      }
    };
  }, []);

  useEffect(() => {
    if (isOnline) {
      notificationInterval.current = setInterval(checkNotifications, 3000);
    } else {
      if (notificationInterval.current) {
        clearInterval(notificationInterval.current);
      }
    }
    return () => {
      if (notificationInterval.current) {
        clearInterval(notificationInterval.current);
      }
    };
  }, [isOnline]);

  const loadEntregadorData = async () => {
    try {
      const data = await AsyncStorage.getItem('entregadorData');
      if (data) {
        const parsed = JSON.parse(data);
        setEntregador(parsed);
        setIsOnline(parsed.status === 'disponivel');
      }
    } catch (error) {
      console.error('Erro ao carregar dados:', error);
    }
  };

  const loadData = async () => {
    try {
      const response = await entregadorAPI.dashboard();
      if (response.data.success) {
        setEntregaAtual(response.data.entregaAtual);
        setEntregasDisponiveis(response.data.entregasDisponiveis || []);
        setGanhosHoje(response.data.ganhosHoje || 0);
        setIsOnline(response.data.status === 'disponivel');
      }
    } catch (error) {
      console.error('Erro ao carregar dados:', error);
    }
  };

  const startLocationTracking = async () => {
    try {
      const { status } = await Location.requestForegroundPermissionsAsync();
      if (status !== 'granted') return;

      Location.watchPositionAsync(
        { accuracy: Location.Accuracy.High, distanceInterval: 50 },
        (location) => {
          entregadorAPI.atualizarLocalizacao(
            location.coords.latitude,
            location.coords.longitude
          );
        }
      );
    } catch (error) {
      console.error('Erro ao rastrear localização:', error);
    }
  };

  const checkNotifications = async () => {
    try {
      const response = await entregadorAPI.verificarNotificacoes();
      if (response.data.novas && response.data.novas.length > 0) {
        playNotificationSound();
        Haptics.notificationAsync(Haptics.NotificationFeedbackType.Success);
        loadData();
      }
    } catch (error) {
      console.error('Erro ao verificar notificações:', error);
    }
  };

  const playNotificationSound = async () => {
    try {
      const { sound } = await Audio.Sound.createAsync(
        require('../../../assets/sounds/notification.mp3')
      );
      soundRef.current = sound;
      await sound.playAsync();
    } catch (error) {
      // Fallback: vibração
      Haptics.notificationAsync(Haptics.NotificationFeedbackType.Warning);
    }
  };

  const toggleOnline = async () => {
    const novoStatus = isOnline ? 'offline' : 'disponivel';
    try {
      const response = await entregadorAPI.atualizarStatus(novoStatus);
      if (response.data.success) {
        setIsOnline(!isOnline);
        Alert.alert(
          isOnline ? '👋 Você desconectou!' : '✅ Você está online!',
          isOnline ? 'Você não receberá novas entregas.' : 'Aguarde novas entregas!'
        );
        loadData();
      }
    } catch (error) {
      Alert.alert('Erro', 'Não foi possível atualizar status');
    }
  };

  const aceitarEntrega = async (entregaId) => {
    try {
      const response = await entregadorAPI.aceitarEntrega(entregaId);
      if (response.data.success) {
        Alert.alert('Sucesso! 🎉', 'Entrega aceita! Vá até o local de coleta.');
        loadData();
      } else {
        Alert.alert('Erro', response.data.message || 'Não foi possível aceitar');
      }
    } catch (error) {
      Alert.alert('Erro', 'Erro de conexão');
    }
  };

  const atualizarStatusEntrega = async (status) => {
    try {
      const response = await entregadorAPI.atualizarEntrega(entregaAtual.id, status);
      if (response.data.success) {
        if (status === 'entregue' && response.data.ganho) {
          const g = response.data.ganho;
          Alert.alert(
            '🎉 Entrega concluída!',
            `💰 Resumo do seu ganho:\n\n💵 Valor bruto: R$ ${g.valor_bruto.toFixed(2)}\n📉 Taxa (${g.taxa_percentual}%): R$ ${g.valor_taxa.toFixed(2)}\n✅ Valor líquido: R$ ${g.valor_liquido.toFixed(2)}`
          );
        }
        loadData();
      }
    } catch (error) {
      Alert.alert('Erro', 'Erro ao atualizar status');
    }
  };

  const cancelarEntrega = async () => {
    if (!motivoSelecionado) {
      Alert.alert('Erro', 'Selecione um motivo');
      return;
    }
    try {
      const response = await entregadorAPI.cancelarEntrega(entregaAtual.id, motivoSelecionado.motivo);
      if (response.data.success) {
        Alert.alert('Entrega cancelada', 'O cliente foi notificado.');
        setShowCancelModal(false);
        setMotivoSelecionado(null);
        loadData();
      }
    } catch (error) {
      Alert.alert('Erro', 'Não foi possível cancelar');
    }
  };

  const abrirNavegacao = (endereco, app) => {
    const encoded = encodeURIComponent(endereco);
    const url = app === 'waze' 
      ? `https://waze.com/ul?q=${encoded}&navigate=yes`
      : `https://www.google.com/maps/search/?api=1&query=${encoded}`;
    Linking.openURL(url);
  };

  const onRefresh = async () => {
    setRefreshing(true);
    await loadData();
    setRefreshing(false);
  };


  return (
    <SafeAreaView style={styles.container}>
      {/* Header */}
      <View style={styles.header}>
        <View style={styles.headerLeft}>
          <View style={[styles.statusDot, { backgroundColor: isOnline ? colors.success : colors.danger }]} />
          <Text style={styles.headerTitle}>Já EntreGo</Text>
        </View>
        <Text style={styles.headerName}>{entregador?.nome?.split(' ')[0] || 'Entregador'}</Text>
      </View>

      <ScrollView
        style={styles.content}
        refreshControl={<RefreshControl refreshing={refreshing} onRefresh={onRefresh} colors={[colors.entregador.primary]} />}
      >
        {/* Toggle Online */}
        <View style={styles.statusToggle}>
          <Text style={styles.statusLabel}>
            Status: <Text style={{ fontWeight: 'bold' }}>{isOnline ? 'Online' : 'Offline'}</Text>
          </Text>
          <TouchableOpacity
            style={[styles.statusBtn, isOnline ? styles.statusBtnOnline : styles.statusBtnOffline]}
            onPress={toggleOnline}
          >
            <Text style={styles.statusBtnText}>{isOnline ? '🟢 Online' : '🔴 Offline'}</Text>
          </TouchableOpacity>
        </View>

        {entregaAtual ? (
          /* Entrega em Andamento */
          <View style={styles.entregaAtualCard}>
            <View style={styles.entregaAtualHeader}>
              <Text style={styles.entregaAtualTitle}>Entrega em Andamento</Text>
              <Text style={styles.entregaAtualStatus}>
                {entregaAtual.status === 'aceita' ? '📦 Ir para Coleta' : '🚚 Ir para Entrega'}
              </Text>
            </View>

            {/* Destino Atual */}
            <View style={styles.destinoAtual}>
              <Text style={styles.destinoIcon}>{entregaAtual.status === 'aceita' ? '📦' : '🏠'}</Text>
              <View style={styles.destinoInfo}>
                <Text style={styles.destinoLabel}>
                  {entregaAtual.status === 'aceita' ? 'Ir para coleta' : 'Ir para entrega'}
                </Text>
                <Text style={styles.destinoEndereco} numberOfLines={2}>
                  {entregaAtual.status === 'aceita' ? entregaAtual.origem_endereco : entregaAtual.destino_endereco}
                </Text>
              </View>
            </View>

            {/* Botões de Navegação */}
            <View style={styles.navButtons}>
              <TouchableOpacity
                style={[styles.navBtn, styles.navBtnWaze]}
                onPress={() => abrirNavegacao(
                  entregaAtual.status === 'aceita' ? entregaAtual.origem_endereco : entregaAtual.destino_endereco,
                  'waze'
                )}
              >
                <Text style={styles.navBtnText}>🗺️ Waze</Text>
              </TouchableOpacity>
              <TouchableOpacity
                style={[styles.navBtn, styles.navBtnGmaps]}
                onPress={() => abrirNavegacao(
                  entregaAtual.status === 'aceita' ? entregaAtual.origem_endereco : entregaAtual.destino_endereco,
                  'gmaps'
                )}
              >
                <Text style={styles.navBtnText}>📍 Google Maps</Text>
              </TouchableOpacity>
            </View>

            {/* Valor */}
            <View style={styles.valorContainer}>
              <Text style={styles.valorLabel}>Valor Total</Text>
              <Text style={styles.valorTexto}>R$ {parseFloat(entregaAtual.valor).toFixed(2).replace('.', ',')}</Text>
              {entregaAtual.distancia && (
                <Text style={styles.valorMeta}>📏 {parseFloat(entregaAtual.distancia).toFixed(1)} km</Text>
              )}
            </View>

            {/* Botões de Ação */}
            <View style={styles.acoesContainer}>
              {entregaAtual.status === 'aceita' ? (
                <>
                  <TouchableOpacity
                    style={styles.btnColetar}
                    onPress={() => atualizarStatusEntrega('coletada')}
                  >
                    <Text style={styles.btnAcaoText}>✓ Confirmar Coleta</Text>
                  </TouchableOpacity>
                  <TouchableOpacity
                    style={styles.btnCancelar}
                    onPress={() => setShowCancelModal(true)}
                  >
                    <Text style={styles.btnCancelarText}>✕ Cancelar Entrega</Text>
                  </TouchableOpacity>
                </>
              ) : (
                <TouchableOpacity
                  style={styles.btnEntregar}
                  onPress={() => atualizarStatusEntrega('entregue')}
                >
                  <Text style={styles.btnAcaoText}>✓ Confirmar Entrega</Text>
                </TouchableOpacity>
              )}
            </View>
          </View>
        ) : (
          <>
            {/* Ganhos do Dia */}
            <View style={styles.ganhosCard}>
              <Text style={styles.ganhosLabel}>Ganhos Líquidos de Hoje</Text>
              <Text style={styles.ganhosValor}>R$ {parseFloat(ganhosHoje).toFixed(2).replace('.', ',')}</Text>
              <Text style={styles.ganhosPeriodo}>{new Date().toLocaleDateString('pt-BR')}</Text>
            </View>

            {/* Entregas Disponíveis */}
            {isOnline ? (
              entregasDisponiveis.length === 0 ? (
                <View style={styles.emptyState}>
                  <Text style={styles.emptyIcon}>📭</Text>
                  <Text style={styles.emptyTitle}>Nenhuma entrega disponível</Text>
                  <Text style={styles.emptySubtitle}>Aguarde, novas entregas aparecerão aqui</Text>
                </View>
              ) : (
                entregasDisponiveis.map((entrega) => (
                  <View key={entrega.id} style={styles.entregaCard}>
                    <View style={styles.entregaCardHeader}>
                      <Text style={styles.entregaCardValor}>
                        R$ {parseFloat(entrega.valor).toFixed(2).replace('.', ',')}
                      </Text>
                      {entrega.distancia && (
                        <Text style={styles.entregaCardMeta}>📏 {parseFloat(entrega.distancia).toFixed(1)} km</Text>
                      )}
                    </View>
                    
                    <View style={styles.rotaContainer}>
                      <View style={styles.rotaPonto}>
                        <View style={[styles.rotaIcon, { backgroundColor: colors.success }]}>
                          <Text style={styles.rotaIconText}>A</Text>
                        </View>
                        <Text style={styles.rotaEndereco} numberOfLines={1}>{entrega.origem_endereco}</Text>
                      </View>
                      <View style={styles.rotaLinha} />
                      <View style={styles.rotaPonto}>
                        <View style={[styles.rotaIcon, { backgroundColor: colors.danger }]}>
                          <Text style={styles.rotaIconText}>B</Text>
                        </View>
                        <Text style={styles.rotaEndereco} numberOfLines={1}>{entrega.destino_endereco}</Text>
                      </View>
                    </View>

                    <TouchableOpacity
                      style={styles.btnAceitar}
                      onPress={() => aceitarEntrega(entrega.id)}
                    >
                      <Text style={styles.btnAceitarText}>Aceitar</Text>
                    </TouchableOpacity>
                  </View>
                ))
              )
            ) : (
              <View style={styles.emptyState}>
                <Text style={styles.emptyIcon}>😴</Text>
                <Text style={styles.emptyTitle}>Você está offline</Text>
                <Text style={styles.emptySubtitle}>Ative o status para receber entregas</Text>
              </View>
            )}
          </>
        )}
      </ScrollView>

      {/* Modal de Cancelamento */}
      <Modal visible={showCancelModal} transparent animationType="slide">
        <View style={styles.modalOverlay}>
          <View style={styles.modalContent}>
            <View style={styles.modalHeader}>
              <Text style={styles.modalTitle}>⚠️ Cancelar Entrega</Text>
              <TouchableOpacity onPress={() => setShowCancelModal(false)}>
                <Text style={styles.modalClose}>✕</Text>
              </TouchableOpacity>
            </View>
            <Text style={styles.modalSubtitle}>Selecione o motivo do cancelamento:</Text>
            
            <ScrollView style={styles.motivosList}>
              {MOTIVOS_CANCELAMENTO.map((m) => (
                <TouchableOpacity
                  key={m.id}
                  style={[styles.motivoItem, motivoSelecionado?.id === m.id && styles.motivoItemSelected]}
                  onPress={() => setMotivoSelecionado(m)}
                >
                  <Text style={styles.motivoIcon}>{m.icon}</Text>
                  <Text style={styles.motivoTexto}>{m.texto}</Text>
                  {motivoSelecionado?.id === m.id && <Text style={styles.motivoCheck}>✓</Text>}
                </TouchableOpacity>
              ))}
            </ScrollView>

            <View style={styles.modalFooter}>
              <TouchableOpacity style={styles.modalBtnVoltar} onPress={() => setShowCancelModal(false)}>
                <Text style={styles.modalBtnVoltarText}>Voltar</Text>
              </TouchableOpacity>
              <TouchableOpacity style={styles.modalBtnConfirmar} onPress={cancelarEntrega}>
                <Text style={styles.modalBtnConfirmarText}>Cancelar Entrega</Text>
              </TouchableOpacity>
            </View>
          </View>
        </View>
      </Modal>
    </SafeAreaView>
  );
}


const styles = StyleSheet.create({
  container: { flex: 1, backgroundColor: colors.background },
  header: {
    flexDirection: 'row', justifyContent: 'space-between', alignItems: 'center',
    backgroundColor: colors.entregador.primary, paddingHorizontal: spacing.lg,
    paddingVertical: spacing.md, paddingTop: spacing.xl,
  },
  headerLeft: { flexDirection: 'row', alignItems: 'center' },
  statusDot: { width: 10, height: 10, borderRadius: 5, marginRight: spacing.sm },
  headerTitle: { fontSize: fonts.sizes.xl, fontWeight: 'bold', color: colors.white },
  headerName: { fontSize: fonts.sizes.md, color: 'rgba(255,255,255,0.8)' },
  content: { flex: 1, padding: spacing.lg },
  statusToggle: {
    flexDirection: 'row', justifyContent: 'space-between', alignItems: 'center',
    backgroundColor: colors.white, borderRadius: borderRadius.lg, padding: spacing.md,
    marginBottom: spacing.lg, ...shadows.sm,
  },
  statusLabel: { fontSize: fonts.sizes.md, color: colors.gray[700] },
  statusBtn: { paddingHorizontal: spacing.lg, paddingVertical: spacing.sm, borderRadius: borderRadius.full },
  statusBtnOnline: { backgroundColor: colors.success },
  statusBtnOffline: { backgroundColor: colors.danger },
  statusBtnText: { color: colors.white, fontWeight: '600' },
  ganhosCard: {
    backgroundColor: colors.entregador.primary, borderRadius: borderRadius.xl,
    padding: spacing.xl, alignItems: 'center', marginBottom: spacing.lg,
  },
  ganhosLabel: { fontSize: fonts.sizes.sm, color: 'rgba(255,255,255,0.8)' },
  ganhosValor: { fontSize: 32, fontWeight: 'bold', color: colors.white, marginVertical: spacing.xs },
  ganhosPeriodo: { fontSize: fonts.sizes.xs, color: 'rgba(255,255,255,0.6)' },
  emptyState: {
    backgroundColor: colors.white, borderRadius: borderRadius.xl, padding: spacing.xxl,
    alignItems: 'center', ...shadows.sm,
  },
  emptyIcon: { fontSize: 48, marginBottom: spacing.md },
  emptyTitle: { fontSize: fonts.sizes.lg, fontWeight: '600', color: colors.gray[700] },
  emptySubtitle: { fontSize: fonts.sizes.sm, color: colors.gray[500], marginTop: spacing.xs },
  entregaCard: {
    backgroundColor: colors.white, borderRadius: borderRadius.xl, padding: spacing.lg,
    marginBottom: spacing.md, ...shadows.sm,
  },
  entregaCardHeader: { flexDirection: 'row', justifyContent: 'space-between', marginBottom: spacing.md },
  entregaCardValor: { fontSize: fonts.sizes.xl, fontWeight: 'bold', color: colors.success },
  entregaCardMeta: { fontSize: fonts.sizes.sm, color: colors.gray[500] },
  rotaContainer: { marginBottom: spacing.md },
  rotaPonto: { flexDirection: 'row', alignItems: 'center' },
  rotaIcon: { width: 24, height: 24, borderRadius: 12, justifyContent: 'center', alignItems: 'center' },
  rotaIconText: { color: colors.white, fontSize: fonts.sizes.xs, fontWeight: 'bold' },
  rotaEndereco: { flex: 1, marginLeft: spacing.sm, fontSize: fonts.sizes.sm, color: colors.gray[700] },
  rotaLinha: { width: 2, height: 16, backgroundColor: colors.gray[300], marginLeft: 11, marginVertical: 4 },
  btnAceitar: {
    backgroundColor: colors.success, borderRadius: borderRadius.lg, padding: spacing.md, alignItems: 'center',
  },
  btnAceitarText: { color: colors.white, fontWeight: 'bold', fontSize: fonts.sizes.md },
  entregaAtualCard: {
    backgroundColor: colors.white, borderRadius: borderRadius.xl, padding: spacing.lg, ...shadows.md,
  },
  entregaAtualHeader: { marginBottom: spacing.md },
  entregaAtualTitle: { fontSize: fonts.sizes.lg, fontWeight: 'bold', color: colors.gray[900] },
  entregaAtualStatus: { fontSize: fonts.sizes.sm, color: colors.entregador.primary, marginTop: 4 },
  destinoAtual: {
    flexDirection: 'row', alignItems: 'center', backgroundColor: colors.entregador.primary,
    borderRadius: borderRadius.lg, padding: spacing.md, marginBottom: spacing.md,
  },
  destinoIcon: { fontSize: 24, marginRight: spacing.sm },
  destinoInfo: { flex: 1 },
  destinoLabel: { fontSize: fonts.sizes.xs, color: 'rgba(255,255,255,0.8)' },
  destinoEndereco: { fontSize: fonts.sizes.sm, color: colors.white, fontWeight: '500' },
  navButtons: { flexDirection: 'row', gap: spacing.sm, marginBottom: spacing.md },
  navBtn: { flex: 1, padding: spacing.md, borderRadius: borderRadius.lg, alignItems: 'center' },
  navBtnWaze: { backgroundColor: '#33ccff' },
  navBtnGmaps: { backgroundColor: '#4285f4' },
  navBtnText: { color: colors.white, fontWeight: '600' },
  valorContainer: {
    backgroundColor: colors.gray[100], borderRadius: borderRadius.lg, padding: spacing.md,
    alignItems: 'center', marginBottom: spacing.md,
  },
  valorLabel: { fontSize: fonts.sizes.xs, color: colors.gray[500] },
  valorTexto: { fontSize: 28, fontWeight: 'bold', color: colors.success },
  valorMeta: { fontSize: fonts.sizes.sm, color: colors.gray[600], marginTop: 4 },
  acoesContainer: { gap: spacing.sm },
  btnColetar: { backgroundColor: colors.success, borderRadius: borderRadius.lg, padding: spacing.md, alignItems: 'center' },
  btnEntregar: { backgroundColor: colors.primary, borderRadius: borderRadius.lg, padding: spacing.md, alignItems: 'center' },
  btnCancelar: { backgroundColor: colors.danger, borderRadius: borderRadius.lg, padding: spacing.md, alignItems: 'center' },
  btnAcaoText: { color: colors.white, fontWeight: 'bold', fontSize: fonts.sizes.md },
  btnCancelarText: { color: colors.white, fontWeight: '600', fontSize: fonts.sizes.md },
  modalOverlay: { flex: 1, backgroundColor: 'rgba(0,0,0,0.5)', justifyContent: 'flex-end' },
  modalContent: {
    backgroundColor: colors.white, borderTopLeftRadius: borderRadius.xl,
    borderTopRightRadius: borderRadius.xl, maxHeight: '80%',
  },
  modalHeader: {
    flexDirection: 'row', justifyContent: 'space-between', alignItems: 'center',
    padding: spacing.lg, borderBottomWidth: 1, borderBottomColor: colors.gray[200],
  },
  modalTitle: { fontSize: fonts.sizes.lg, fontWeight: 'bold', color: colors.danger },
  modalClose: { fontSize: 24, color: colors.gray[500] },
  modalSubtitle: { padding: spacing.lg, paddingBottom: spacing.sm, color: colors.gray[600] },
  motivosList: { paddingHorizontal: spacing.lg },
  motivoItem: {
    flexDirection: 'row', alignItems: 'center', padding: spacing.md,
    borderRadius: borderRadius.lg, borderWidth: 2, borderColor: colors.gray[200], marginBottom: spacing.sm,
  },
  motivoItemSelected: { borderColor: colors.danger, backgroundColor: '#fff5f5' },
  motivoIcon: { fontSize: 20, marginRight: spacing.md },
  motivoTexto: { flex: 1, fontSize: fonts.sizes.md, color: colors.gray[800] },
  motivoCheck: { color: colors.danger, fontWeight: 'bold' },
  modalFooter: { flexDirection: 'row', padding: spacing.lg, gap: spacing.sm },
  modalBtnVoltar: { flex: 1, backgroundColor: colors.gray[200], borderRadius: borderRadius.lg, padding: spacing.md, alignItems: 'center' },
  modalBtnVoltarText: { color: colors.gray[700], fontWeight: '600' },
  modalBtnConfirmar: { flex: 1, backgroundColor: colors.danger, borderRadius: borderRadius.lg, padding: spacing.md, alignItems: 'center' },
  modalBtnConfirmarText: { color: colors.white, fontWeight: '600' },
});