import React, { useState, useEffect } from 'react';
import {
  View,
  Text,
  StyleSheet,
  SafeAreaView,
  ScrollView,
  RefreshControl,
} from 'react-native';
import { colors, fonts, spacing, borderRadius, shadows } from '../../config/theme';
import { entregadorAPI } from '../../config/api';

export default function GanhosScreen() {
  const [ganhos, setGanhos] = useState({
    hoje: { liquido: 0, bruto: 0, taxa: 0, qtd: 0 },
    semana: { liquido: 0, bruto: 0, qtd: 0 },
    mes: { liquido: 0, bruto: 0, qtd: 0 },
    porDia: [],
    detalhados: [],
  });
  const [taxaPercentual, setTaxaPercentual] = useState(15);
  const [refreshing, setRefreshing] = useState(false);

  useEffect(() => {
    loadGanhos();
  }, []);

  const loadGanhos = async () => {
    try {
      const response = await entregadorAPI.ganhos();
      if (response.data.success) {
        setGanhos(response.data);
        setTaxaPercentual(response.data.taxaPercentual || 15);
      }
    } catch (error) {
      console.error('Erro ao carregar ganhos:', error);
    }
  };

  const onRefresh = async () => {
    setRefreshing(true);
    await loadGanhos();
    setRefreshing(false);
  };

  return (
    <SafeAreaView style={styles.container}>
      {/* Header */}
      <View style={styles.header}>
        <Text style={styles.headerTitle}>💰 Meus Ganhos</Text>
      </View>

      <ScrollView
        style={styles.content}
        refreshControl={<RefreshControl refreshing={refreshing} onRefresh={onRefresh} colors={[colors.entregador.primary]} />}
      >
        {/* Info Taxa */}
        <View style={styles.taxaInfo}>
          <Text style={styles.taxaText}>
            💡 Taxa da plataforma: {taxaPercentual}% por entrega
          </Text>
        </View>

        {/* Ganhos Hoje */}
        <View style={styles.ganhosCard}>
          <Text style={styles.ganhosLabel}>Ganho Líquido Hoje</Text>
          <Text style={styles.ganhosValor}>
            R$ {parseFloat(ganhos.hoje?.liquido || 0).toFixed(2).replace('.', ',')}
          </Text>
          <Text style={styles.ganhosPeriodo}>{ganhos.hoje?.qtd || 0} entregas</Text>
          <Text style={styles.ganhosDetalhe}>
            Bruto: R$ {parseFloat(ganhos.hoje?.bruto || 0).toFixed(2).replace('.', ',')} | 
            Taxa: R$ {parseFloat(ganhos.hoje?.taxa || 0).toFixed(2).replace('.', ',')}
          </Text>
        </View>

        {/* Resumo Semana/Mês */}
        <View style={styles.resumoContainer}>
          <View style={styles.resumoCard}>
            <Text style={styles.resumoLabel}>Esta Semana</Text>
            <Text style={styles.resumoValor}>
              R$ {parseFloat(ganhos.semana?.liquido || 0).toFixed(2).replace('.', ',')}
            </Text>
            <Text style={styles.resumoQtd}>{ganhos.semana?.qtd || 0} entregas</Text>
          </View>
          <View style={styles.resumoCard}>
            <Text style={styles.resumoLabel}>Este Mês</Text>
            <Text style={styles.resumoValor}>
              R$ {parseFloat(ganhos.mes?.liquido || 0).toFixed(2).replace('.', ',')}
            </Text>
            <Text style={styles.resumoQtd}>{ganhos.mes?.qtd || 0} entregas</Text>
          </View>
        </View>

        {/* Últimos 7 dias */}
        <View style={styles.section}>
          <Text style={styles.sectionTitle}>Últimos 7 dias</Text>
          {ganhos.porDia?.length === 0 ? (
            <Text style={styles.emptyText}>Nenhum ganho registrado</Text>
          ) : (
            ganhos.porDia?.map((dia, index) => (
              <View key={index} style={styles.diaItem}>
                <View>
                  <Text style={styles.diaData}>{new Date(dia.dia).toLocaleDateString('pt-BR', { day: '2-digit', month: '2-digit' })}</Text>
                  <Text style={styles.diaQtd}>{dia.qtd} entregas</Text>
                </View>
                <View style={styles.diaValores}>
                  <Text style={styles.diaLiquido}>R$ {parseFloat(dia.liquido).toFixed(2).replace('.', ',')}</Text>
                  <Text style={styles.diaBruto}>Bruto: R$ {parseFloat(dia.bruto).toFixed(2).replace('.', ',')}</Text>
                </View>
              </View>
            ))
          )}
        </View>

        {/* Últimas Entregas */}
        <View style={styles.section}>
          <Text style={styles.sectionTitle}>Últimas Entregas</Text>
          {ganhos.detalhados?.length === 0 ? (
            <Text style={styles.emptyText}>Nenhuma entrega concluída ainda</Text>
          ) : (
            ganhos.detalhados?.map((g, index) => (
              <View key={index} style={styles.entregaItem}>
                <View style={styles.entregaInfo}>
                  <Text style={styles.entregaData}>
                    {new Date(g.data_entrega).toLocaleDateString('pt-BR')} {new Date(g.data_entrega).toLocaleTimeString('pt-BR', { hour: '2-digit', minute: '2-digit' })}
                  </Text>
                  <Text style={styles.entregaEndereco} numberOfLines={1}>
                    📍 {g.origem_endereco?.substring(0, 25)}...
                  </Text>
                </View>
                <View style={styles.entregaValores}>
                  <Text style={styles.entregaLiquido}>R$ {parseFloat(g.valor_liquido).toFixed(2).replace('.', ',')}</Text>
                  <Text style={styles.entregaBruto}>Bruto: R$ {parseFloat(g.valor_bruto).toFixed(2).replace('.', ',')}</Text>
                  <Text style={styles.entregaTaxa}>Taxa: R$ {parseFloat(g.valor_taxa).toFixed(2).replace('.', ',')} ({g.taxa_percentual}%)</Text>
                </View>
              </View>
            ))
          )}
        </View>

        <View style={{ height: 40 }} />
      </ScrollView>
    </SafeAreaView>
  );
}

const styles = StyleSheet.create({
  container: { flex: 1, backgroundColor: colors.background },
  header: {
    backgroundColor: colors.entregador.primary, paddingHorizontal: spacing.lg,
    paddingVertical: spacing.lg, paddingTop: spacing.xl,
  },
  headerTitle: { fontSize: fonts.sizes.xxl, fontWeight: 'bold', color: colors.white },
  content: { flex: 1, padding: spacing.lg },
  taxaInfo: {
    backgroundColor: '#fff3e0', padding: spacing.md, borderRadius: borderRadius.lg, marginBottom: spacing.lg,
  },
  taxaText: { fontSize: fonts.sizes.sm, color: '#e65100' },
  ganhosCard: {
    backgroundColor: colors.entregador.primary, borderRadius: borderRadius.xl,
    padding: spacing.xl, alignItems: 'center', marginBottom: spacing.lg,
  },
  ganhosLabel: { fontSize: fonts.sizes.sm, color: 'rgba(255,255,255,0.8)' },
  ganhosValor: { fontSize: 36, fontWeight: 'bold', color: colors.white, marginVertical: spacing.xs },
  ganhosPeriodo: { fontSize: fonts.sizes.sm, color: 'rgba(255,255,255,0.7)' },
  ganhosDetalhe: { fontSize: fonts.sizes.xs, color: 'rgba(255,255,255,0.6)', marginTop: spacing.sm },
  resumoContainer: { flexDirection: 'row', gap: spacing.md, marginBottom: spacing.lg },
  resumoCard: {
    flex: 1, backgroundColor: colors.white, borderRadius: borderRadius.xl,
    padding: spacing.lg, alignItems: 'center', ...shadows.sm,
  },
  resumoLabel: { fontSize: fonts.sizes.xs, color: colors.gray[600] },
  resumoValor: { fontSize: fonts.sizes.xl, fontWeight: 'bold', color: colors.entregador.primary, marginVertical: 4 },
  resumoQtd: { fontSize: fonts.sizes.xs, color: colors.gray[500] },
  section: {
    backgroundColor: colors.white, borderRadius: borderRadius.xl, padding: spacing.lg,
    marginBottom: spacing.lg, ...shadows.sm,
  },
  sectionTitle: { fontSize: fonts.sizes.lg, fontWeight: 'bold', color: colors.gray[900], marginBottom: spacing.md },
  emptyText: { textAlign: 'center', color: colors.gray[500], paddingVertical: spacing.lg },
  diaItem: {
    flexDirection: 'row', justifyContent: 'space-between', alignItems: 'center',
    paddingVertical: spacing.md, borderBottomWidth: 1, borderBottomColor: colors.gray[100],
  },
  diaData: { fontSize: fonts.sizes.md, fontWeight: '500', color: colors.gray[800] },
  diaQtd: { fontSize: fonts.sizes.xs, color: colors.gray[500] },
  diaValores: { alignItems: 'flex-end' },
  diaLiquido: { fontSize: fonts.sizes.md, fontWeight: 'bold', color: colors.success },
  diaBruto: { fontSize: fonts.sizes.xs, color: colors.gray[500] },
  entregaItem: {
    flexDirection: 'row', justifyContent: 'space-between', paddingVertical: spacing.md,
    borderBottomWidth: 1, borderBottomColor: colors.gray[100],
  },
  entregaInfo: { flex: 1 },
  entregaData: { fontSize: fonts.sizes.xs, color: colors.gray[500] },
  entregaEndereco: { fontSize: fonts.sizes.sm, color: colors.gray[700], marginTop: 2 },
  entregaValores: { alignItems: 'flex-end' },
  entregaLiquido: { fontSize: fonts.sizes.md, fontWeight: 'bold', color: colors.success },
  entregaBruto: { fontSize: fonts.sizes.xs, color: colors.gray[500] },
  entregaTaxa: { fontSize: fonts.sizes.xs, color: colors.danger },
});