import React, { useState, useEffect } from 'react';
import {
  View,
  Text,
  StyleSheet,
  SafeAreaView,
  ScrollView,
  TouchableOpacity,
  Alert,
} from 'react-native';
import AsyncStorage from '@react-native-async-storage/async-storage';
import { colors, fonts, spacing, borderRadius, shadows } from '../../config/theme';

export default function PerfilScreen({ navigation, setIsLoggedIn }) {
  const [entregador, setEntregador] = useState(null);

  useEffect(() => {
    loadEntregadorData();
  }, []);

  const loadEntregadorData = async () => {
    try {
      const data = await AsyncStorage.getItem('entregadorData');
      if (data) {
        setEntregador(JSON.parse(data));
      }
    } catch (error) {
      console.error('Erro ao carregar dados:', error);
    }
  };

  const handleLogout = () => {
    Alert.alert(
      'Sair',
      'Deseja realmente sair da sua conta?',
      [
        { text: 'Cancelar', style: 'cancel' },
        {
          text: 'Sair',
          style: 'destructive',
          onPress: async () => {
            await AsyncStorage.removeItem('entregadorToken');
            await AsyncStorage.removeItem('entregadorData');
            setIsLoggedIn(false);
          },
        },
      ]
    );
  };

  const getVeiculoIcon = (veiculo) => {
    switch (veiculo) {
      case 'moto': return '🏍️';
      case 'carro': return '🚗';
      case 'van': return '🚐';
      case 'caminhao': return '🚛';
      default: return '🚗';
    }
  };

  const menuItems = [
    { icon: '👤', title: 'Editar Perfil', subtitle: 'Altere seus dados pessoais' },
    { icon: '🚗', title: 'Meu Veículo', subtitle: 'Dados do veículo' },
    { icon: '📄', title: 'Documentos', subtitle: 'Envie seus documentos' },
    { icon: '🔔', title: 'Notificações', subtitle: 'Configure seus alertas' },
    { icon: '💳', title: 'Dados Bancários', subtitle: 'Para receber pagamentos' },
    { icon: '❓', title: 'Ajuda', subtitle: 'Central de suporte' },
    { icon: '📄', title: 'Termos de Uso', subtitle: 'Políticas e termos' },
  ];

  return (
    <SafeAreaView style={styles.container}>
      {/* Header */}
      <View style={styles.header}>
        <View style={styles.avatarContainer}>
          <Text style={styles.avatar}>{getVeiculoIcon(entregador?.veiculo)}</Text>
        </View>
        <Text style={styles.nome}>{entregador?.nome || 'Entregador'}</Text>
        <Text style={styles.email}>{entregador?.email || ''}</Text>
        {entregador?.veiculo && (
          <View style={styles.veiculoBadge}>
            <Text style={styles.veiculoText}>
              {getVeiculoIcon(entregador.veiculo)} {entregador.veiculo?.charAt(0).toUpperCase() + entregador.veiculo?.slice(1)}
              {entregador.placa && ` • ${entregador.placa}`}
            </Text>
          </View>
        )}
      </View>

      <ScrollView style={styles.content}>
        {/* Menu Items */}
        <View style={styles.menuContainer}>
          {menuItems.map((item, index) => (
            <TouchableOpacity key={index} style={styles.menuItem}>
              <Text style={styles.menuIcon}>{item.icon}</Text>
              <View style={styles.menuText}>
                <Text style={styles.menuTitle}>{item.title}</Text>
                <Text style={styles.menuSubtitle}>{item.subtitle}</Text>
              </View>
              <Text style={styles.menuArrow}>›</Text>
            </TouchableOpacity>
          ))}
        </View>

        {/* Botão Sair */}
        <TouchableOpacity style={styles.logoutBtn} onPress={handleLogout}>
          <Text style={styles.logoutIcon}>🚪</Text>
          <Text style={styles.logoutText}>Sair da Conta</Text>
        </TouchableOpacity>

        {/* Versão */}
        <Text style={styles.version}>Já EntreGo Entregador v1.0.0</Text>
      </ScrollView>
    </SafeAreaView>
  );
}

const styles = StyleSheet.create({
  container: { flex: 1, backgroundColor: colors.background },
  header: {
    backgroundColor: colors.entregador.primary, paddingHorizontal: spacing.lg,
    paddingVertical: spacing.xxl, alignItems: 'center',
  },
  avatarContainer: {
    width: 80, height: 80, borderRadius: 40, backgroundColor: colors.white,
    justifyContent: 'center', alignItems: 'center', marginBottom: spacing.md,
  },
  avatar: { fontSize: 40 },
  nome: { fontSize: fonts.sizes.xl, fontWeight: 'bold', color: colors.white },
  email: { fontSize: fonts.sizes.sm, color: 'rgba(255,255,255,0.8)', marginTop: 4 },
  veiculoBadge: {
    backgroundColor: 'rgba(255,255,255,0.2)', paddingHorizontal: spacing.md,
    paddingVertical: spacing.xs, borderRadius: borderRadius.full, marginTop: spacing.sm,
  },
  veiculoText: { color: colors.white, fontSize: fonts.sizes.sm },
  content: { flex: 1, padding: spacing.lg },
  menuContainer: { backgroundColor: colors.white, borderRadius: borderRadius.xl, ...shadows.sm },
  menuItem: {
    flexDirection: 'row', alignItems: 'center', padding: spacing.lg,
    borderBottomWidth: 1, borderBottomColor: colors.gray[100],
  },
  menuIcon: { fontSize: 24, marginRight: spacing.md },
  menuText: { flex: 1 },
  menuTitle: { fontSize: fonts.sizes.md, fontWeight: '600', color: colors.gray[900] },
  menuSubtitle: { fontSize: fonts.sizes.xs, color: colors.gray[500], marginTop: 2 },
  menuArrow: { fontSize: 24, color: colors.gray[400] },
  logoutBtn: {
    flexDirection: 'row', alignItems: 'center', justifyContent: 'center',
    backgroundColor: colors.white, borderRadius: borderRadius.xl, padding: spacing.lg,
    marginTop: spacing.xl, borderWidth: 1, borderColor: colors.danger,
  },
  logoutIcon: { fontSize: 20, marginRight: spacing.sm },
  logoutText: { fontSize: fonts.sizes.md, fontWeight: '600', color: colors.danger },
  version: {
    textAlign: 'center', color: colors.gray[400], fontSize: fonts.sizes.xs,
    marginTop: spacing.xl, marginBottom: spacing.xxl,
  },
});