<?php
require_once '../config/database.php';
require_once '../config/auth.php';
requireLogin('admin', 'index.php');

$db = Database::getInstance()->getConnection();
$sucesso = '';
$erro = '';

// Salvar configurações
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    try {
        foreach ($_POST as $chave => $valor) {
            if (strpos($chave, 'config_') === 0) {
                $chaveReal = substr($chave, 7);
                $stmt = $db->prepare("INSERT INTO configuracoes (chave, valor) VALUES (?, ?) ON DUPLICATE KEY UPDATE valor = ?");
                $stmt->execute([$chaveReal, $valor, $valor]);
            }
        }
        $sucesso = 'Configurações salvas com sucesso!';
    } catch (Exception $e) {
        $erro = 'Erro ao salvar: ' . $e->getMessage();
    }
}

// Buscar configurações
$stmt = $db->query("SELECT * FROM configuracoes");
$configsRaw = $stmt->fetchAll(PDO::FETCH_ASSOC);
$configs = [];
foreach ($configsRaw as $cfg) {
    $configs[$cfg['chave']] = $cfg['valor'];
}

// Valores padrão para categorias
$categorias = [
    'moto' => ['nome' => 'Moto', 'icon' => '🏍️', 'desc' => 'Entregas rápidas e pequenas'],
    'carro' => ['nome' => 'Carro', 'icon' => '🚗', 'desc' => 'Pacotes médios'],
    'van' => ['nome' => 'Van', 'icon' => '🚐', 'desc' => 'Volumes maiores'],
    'caminhao' => ['nome' => 'Caminhão', 'icon' => '🚛', 'desc' => 'Cargas pesadas']
];

$pageTitle = 'Configurações';
include 'includes/header.php';
?>
    
    <div class="container">
        <h2 style="margin-bottom: 20px;">⚙️ Configurações do Sistema</h2>
        
        <?php if ($sucesso): ?>
            <div class="alert alert-success"><?= htmlspecialchars($sucesso) ?></div>
        <?php endif; ?>
        
        <?php if ($erro): ?>
            <div class="alert alert-danger"><?= htmlspecialchars($erro) ?></div>
        <?php endif; ?>
        
        <form method="POST">
            <!-- Preços por Categoria de Veículo -->
            <div class="config-card">
                <h3>🚚 Preços por Categoria de Veículo</h3>
                <p style="color: var(--text-muted); margin-bottom: 20px;">
                    Configure o valor base e o valor por km para cada tipo de veículo.
                </p>
                
                <div class="categorias-grid">
                    <?php foreach ($categorias as $key => $cat): ?>
                    <div class="categoria-card">
                        <div class="categoria-header">
                            <span class="categoria-icon"><?= $cat['icon'] ?></span>
                            <div>
                                <strong><?= $cat['nome'] ?></strong>
                                <small><?= $cat['desc'] ?></small>
                            </div>
                        </div>
                        <div class="categoria-inputs">
                            <div class="input-row">
                                <label>Taxa Base</label>
                                <div class="input-group">
                                    <span>R$</span>
                                    <input type="number" 
                                           name="config_preco_base_<?= $key ?>" 
                                           value="<?= htmlspecialchars($configs["preco_base_$key"] ?? ($key == 'moto' ? '8' : ($key == 'carro' ? '12' : ($key == 'van' ? '20' : '35')))) ?>" 
                                           class="config-input"
                                           step="0.50"
                                           min="0">
                                </div>
                            </div>
                            <div class="input-row">
                                <label>Por Km</label>
                                <div class="input-group">
                                    <span>R$</span>
                                    <input type="number" 
                                           name="config_preco_km_<?= $key ?>" 
                                           value="<?= htmlspecialchars($configs["preco_km_$key"] ?? ($key == 'moto' ? '2.00' : ($key == 'carro' ? '2.50' : ($key == 'van' ? '3.50' : '5.00')))) ?>" 
                                           class="config-input"
                                           step="0.10"
                                           min="0">
                                </div>
                            </div>
                        </div>
                        <div class="categoria-exemplo">
                            <small>Exemplo 10km:</small>
                            <strong id="exemplo-<?= $key ?>">R$ 0,00</strong>
                        </div>
                    </div>
                    <?php endforeach; ?>
                </div>
            </div>
            
            <!-- Taxas do Entregador -->
            <div class="config-card">
                <h3>💰 Taxas da Plataforma (Entregador)</h3>
                <p style="color: var(--text-muted); margin-bottom: 20px;">
                    Configure as taxas que serão descontadas do valor da entrega para o entregador.
                </p>
                
                <div class="config-item">
                    <div class="config-info">
                        <label>Taxa Percentual</label>
                        <small>Porcentagem descontada de cada entrega</small>
                    </div>
                    <div class="input-group">
                        <input type="number" 
                               name="config_taxa_entregador_percentual" 
                               value="<?= htmlspecialchars($configs['taxa_entregador_percentual'] ?? '15') ?>" 
                               class="config-input"
                               step="0.5"
                               min="0"
                               max="50">
                        <span class="input-suffix">%</span>
                    </div>
                </div>
                
                <div class="config-item">
                    <div class="config-info">
                        <label>Taxa Fixa</label>
                        <small>Valor fixo descontado por entrega</small>
                    </div>
                    <div class="input-group">
                        <span class="input-suffix">R$</span>
                        <input type="number" 
                               name="config_taxa_entregador_fixa" 
                               value="<?= htmlspecialchars($configs['taxa_entregador_fixa'] ?? '0') ?>" 
                               class="config-input"
                               step="0.50"
                               min="0">
                    </div>
                </div>
                
                <!-- Preview de cálculo -->
                <div class="preview-box">
                    <h4>📊 Exemplo de Cálculo (Entrega R$ 30,00)</h4>
                    <div class="preview-item">
                        <span>Valor da entrega</span>
                        <span>R$ 30,00</span>
                    </div>
                    <div class="preview-item" style="color: #c62828;">
                        <span>Taxa percentual (<span id="taxa-perc-preview"><?= $configs['taxa_entregador_percentual'] ?? 15 ?></span>%)</span>
                        <span id="taxa-perc-valor">- R$ <?= number_format(30 * (($configs['taxa_entregador_percentual'] ?? 15) / 100), 2, ',', '.') ?></span>
                    </div>
                    <div class="preview-item" style="color: #c62828;">
                        <span>Taxa fixa</span>
                        <span id="taxa-fixa-valor">- R$ <?= number_format($configs['taxa_entregador_fixa'] ?? 0, 2, ',', '.') ?></span>
                    </div>
                    <div class="preview-item total" style="color: #2e7d32;">
                        <span>Entregador recebe</span>
                        <span id="entregador-recebe">R$ <?= number_format(30 - (30 * (($configs['taxa_entregador_percentual'] ?? 15) / 100)) - ($configs['taxa_entregador_fixa'] ?? 0), 2, ',', '.') ?></span>
                    </div>
                </div>
            </div>
            
            <!-- Configurações Gerais -->
            <div class="config-card">
                <h3>🔧 Configurações Gerais</h3>
                
                <div class="config-item">
                    <div class="config-info">
                        <label>Valor Mínimo de Entrega</label>
                        <small>Valor mínimo cobrado independente da distância</small>
                    </div>
                    <div class="input-group">
                        <span class="input-suffix">R$</span>
                        <input type="number" 
                               name="config_valor_minimo_entrega" 
                               value="<?= htmlspecialchars($configs['valor_minimo_entrega'] ?? '10') ?>" 
                               class="config-input"
                               step="0.50"
                               min="0">
                    </div>
                </div>
                
                <div class="config-item">
                    <div class="config-info">
                        <label>Raio Máximo de Entrega</label>
                        <small>Distância máxima permitida para entregas</small>
                    </div>
                    <div class="input-group">
                        <input type="number" 
                               name="config_raio_maximo_km" 
                               value="<?= htmlspecialchars($configs['raio_maximo_km'] ?? '50') ?>" 
                               class="config-input"
                               step="1"
                               min="1">
                        <span class="input-suffix">km</span>
                    </div>
                </div>
            </div>
            
            <button type="submit" class="btn btn-primary" style="padding: 16px 40px; font-size: 16px;">
                💾 Salvar Configurações
            </button>
        </form>
    </div>
    
    <style>
        .config-card {
            background: var(--card-bg);
            border-radius: 12px;
            padding: 24px;
            margin-bottom: 20px;
            box-shadow: 0 1px 3px rgba(0,0,0,0.1);
        }
        .config-card h3 {
            color: var(--primary);
            margin-bottom: 10px;
            padding-bottom: 10px;
            border-bottom: 2px solid var(--border-color);
        }
        .categorias-grid {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(250px, 1fr));
            gap: 20px;
        }
        .categoria-card {
            background: var(--bg-primary);
            border: 2px solid var(--border-color);
            border-radius: 12px;
            padding: 20px;
            transition: all 0.3s;
        }
        .categoria-card:hover {
            border-color: var(--primary);
            box-shadow: 0 4px 12px rgba(0,0,0,0.1);
        }
        .categoria-header {
            display: flex;
            align-items: center;
            gap: 12px;
            margin-bottom: 20px;
        }
        .categoria-icon {
            font-size: 36px;
        }
        .categoria-header strong {
            display: block;
            font-size: 16px;
            color: var(--text-primary);
        }
        .categoria-header small {
            color: var(--text-muted);
            font-size: 12px;
        }
        .categoria-inputs {
            display: flex;
            flex-direction: column;
            gap: 12px;
        }
        .input-row {
            display: flex;
            justify-content: space-between;
            align-items: center;
        }
        .input-row label {
            font-size: 14px;
            color: var(--text-secondary);
        }
        .input-row .input-group {
            display: flex;
            align-items: center;
            gap: 6px;
        }
        .input-row .input-group span {
            color: var(--text-muted);
            font-size: 13px;
        }
        .input-row .config-input {
            width: 80px;
            padding: 8px 10px;
            border: 1px solid var(--border-color);
            border-radius: 6px;
            font-size: 14px;
            text-align: right;
            background: var(--card-bg);
            color: var(--text-primary);
        }
        .categoria-exemplo {
            margin-top: 15px;
            padding-top: 15px;
            border-top: 1px dashed var(--border-color);
            display: flex;
            justify-content: space-between;
            align-items: center;
        }
        .categoria-exemplo small {
            color: var(--text-muted);
        }
        .categoria-exemplo strong {
            color: var(--primary);
            font-size: 18px;
        }
        .config-item {
            display: flex;
            align-items: center;
            justify-content: space-between;
            padding: 16px 0;
            border-bottom: 1px solid var(--border-color);
        }
        .config-item:last-child {
            border-bottom: none;
        }
        .config-info {
            flex: 1;
        }
        .config-info label {
            display: block;
            font-weight: 600;
            color: var(--text-primary);
            margin-bottom: 4px;
        }
        .config-info small {
            color: var(--text-muted);
            font-size: 13px;
        }
        .config-input {
            width: 120px;
            padding: 10px 14px;
            border: 2px solid var(--border-color);
            border-radius: 8px;
            font-size: 16px;
            text-align: right;
            background: var(--card-bg);
            color: var(--text-primary);
        }
        .config-input:focus {
            outline: none;
            border-color: var(--primary);
        }
        .input-group {
            display: flex;
            align-items: center;
            gap: 8px;
        }
        .input-suffix {
            color: var(--text-muted);
            font-weight: 500;
        }
        .preview-box {
            background: var(--bg-primary);
            border-radius: 12px;
            padding: 20px;
            margin-top: 20px;
            border: 1px solid var(--border-color);
        }
        .preview-box h4 {
            color: var(--primary);
            margin-bottom: 15px;
        }
        .preview-item {
            display: flex;
            justify-content: space-between;
            padding: 8px 0;
            color: var(--text-primary);
        }
        .preview-item.total {
            border-top: 2px solid var(--primary);
            margin-top: 10px;
            padding-top: 15px;
            font-weight: bold;
            font-size: 18px;
        }
    </style>
    
    <script>
    // Atualizar exemplos em tempo real
    function atualizarExemplos() {
        const categorias = ['moto', 'carro', 'van', 'caminhao'];
        categorias.forEach(cat => {
            const base = parseFloat(document.querySelector(`[name="config_preco_base_${cat}"]`).value) || 0;
            const km = parseFloat(document.querySelector(`[name="config_preco_km_${cat}"]`).value) || 0;
            const total = base + (km * 10);
            document.getElementById(`exemplo-${cat}`).textContent = 'R$ ' + total.toFixed(2).replace('.', ',');
        });
        
        // Atualizar preview de taxas
        const taxaPerc = parseFloat(document.querySelector('[name="config_taxa_entregador_percentual"]').value) || 0;
        const taxaFixa = parseFloat(document.querySelector('[name="config_taxa_entregador_fixa"]').value) || 0;
        const valorEntrega = 30;
        const descontoPerc = valorEntrega * (taxaPerc / 100);
        const entregadorRecebe = valorEntrega - descontoPerc - taxaFixa;
        
        document.getElementById('taxa-perc-preview').textContent = taxaPerc;
        document.getElementById('taxa-perc-valor').textContent = '- R$ ' + descontoPerc.toFixed(2).replace('.', ',');
        document.getElementById('taxa-fixa-valor').textContent = '- R$ ' + taxaFixa.toFixed(2).replace('.', ',');
        document.getElementById('entregador-recebe').textContent = 'R$ ' + entregadorRecebe.toFixed(2).replace('.', ',');
    }
    
    // Adicionar listeners
    document.querySelectorAll('.config-input').forEach(input => {
        input.addEventListener('input', atualizarExemplos);
    });
    
    // Calcular ao carregar
    atualizarExemplos();
    </script>
</body>
</html>
