// Sistema de notificações para entregadores
class NotificacaoEntregador {
    constructor() {
        this.audioContext = null;
        this.notificacoesVistas = new Set();
        this.intervalo = null;
    }

    init() {
        this.solicitarPermissao();
        this.iniciarVerificacao();
    }

    // Solicitar permissão para notificações do navegador
    solicitarPermissao() {
        if ('Notification' in window && Notification.permission === 'default') {
            Notification.requestPermission();
        }
    }

    // Criar contexto de áudio para tocar som
    criarAudioContext() {
        if (!this.audioContext) {
            this.audioContext = new (window.AudioContext || window.webkitAudioContext)();
        }
        return this.audioContext;
    }

    // Tocar som de notificação
    tocarSom() {
        try {
            const ctx = this.criarAudioContext();
            
            // Criar oscilador para gerar som
            const oscillator = ctx.createOscillator();
            const gainNode = ctx.createGain();
            
            oscillator.connect(gainNode);
            gainNode.connect(ctx.destination);
            
            // Configurar som (tipo sino)
            oscillator.frequency.setValueAtTime(880, ctx.currentTime); // Nota A5
            oscillator.type = 'sine';
            
            // Envelope de volume
            gainNode.gain.setValueAtTime(0, ctx.currentTime);
            gainNode.gain.linearRampToValueAtTime(0.5, ctx.currentTime + 0.1);
            gainNode.gain.linearRampToValueAtTime(0, ctx.currentTime + 0.5);
            
            oscillator.start(ctx.currentTime);
            oscillator.stop(ctx.currentTime + 0.5);
            
            // Segunda nota
            setTimeout(() => {
                const osc2 = ctx.createOscillator();
                const gain2 = ctx.createGain();
                osc2.connect(gain2);
                gain2.connect(ctx.destination);
                osc2.frequency.setValueAtTime(1100, ctx.currentTime);
                osc2.type = 'sine';
                gain2.gain.setValueAtTime(0, ctx.currentTime);
                gain2.gain.linearRampToValueAtTime(0.5, ctx.currentTime + 0.1);
                gain2.gain.linearRampToValueAtTime(0, ctx.currentTime + 0.5);
                osc2.start(ctx.currentTime);
                osc2.stop(ctx.currentTime + 0.5);
            }, 200);
            
        } catch (e) {
            console.log('Erro ao tocar som:', e);
        }
        
        // Vibrar se disponível
        if ('vibrate' in navigator) {
            navigator.vibrate([200, 100, 200, 100, 200]);
        }
    }

    // Mostrar notificação do navegador
    mostrarNotificacaoNavegador(titulo, corpo) {
        if ('Notification' in window && Notification.permission === 'granted') {
            const notif = new Notification(titulo, {
                body: corpo,
                icon: '🚚',
                tag: 'nova-entrega',
                requireInteraction: true
            });
            
            notif.onclick = () => {
                window.focus();
                notif.close();
            };
        }
    }

    // Mostrar notificação visual na página
    mostrarNotificacaoVisual(mensagem) {
        const popup = document.getElementById('notificacao');
        const texto = document.getElementById('notificacao-texto');
        
        if (popup && texto) {
            texto.textContent = mensagem;
            popup.classList.add('show');
            
            setTimeout(() => {
                popup.classList.remove('show');
            }, 5000);
        }
    }

    // Verificar novas notificações
    async verificar() {
        try {
            const response = await fetch('api/verificar-notificacoes.php');
            const data = await response.json();
            
            if (data.novas && data.novas.length > 0) {
                data.novas.forEach(notif => {
                    if (!this.notificacoesVistas.has(notif.id)) {
                        this.notificacoesVistas.add(notif.id);
                        
                        // Tocar som
                        this.tocarSom();
                        
                        // Mostrar notificação visual
                        this.mostrarNotificacaoVisual(notif.mensagem);
                        
                        // Mostrar notificação do navegador
                        this.mostrarNotificacaoNavegador(
                            '🚚 Nova Entrega!',
                            notif.mensagem
                        );
                    }
                });
                
                // Recarregar página após 2 segundos
                setTimeout(() => location.reload(), 2000);
            }
        } catch (e) {
            console.log('Erro ao verificar notificações:', e);
        }
    }

    // Iniciar verificação periódica
    iniciarVerificacao() {
        // Verificar a cada 5 segundos
        this.intervalo = setInterval(() => this.verificar(), 5000);
    }

    // Parar verificação
    pararVerificacao() {
        if (this.intervalo) {
            clearInterval(this.intervalo);
        }
    }
}

// Inicializar quando a página carregar
document.addEventListener('DOMContentLoaded', () => {
    if (document.body.classList.contains('entregador')) {
        window.notificador = new NotificacaoEntregador();
        window.notificador.init();
    }
});
