<?php
require_once '../config/database.php';
require_once '../config/auth.php';
require_once '../config/helpers.php';
requireLogin('usuario', '../index.php');

$db = Database::getInstance()->getConnection();
$userId = getCurrentUserId('usuario');

// Verificar se usuário está bloqueado
$stmt = $db->prepare("SELECT bloqueado, motivo_bloqueio FROM usuarios WHERE id = ?");
$stmt->execute([$userId]);
$usuarioStatus = $stmt->fetch(PDO::FETCH_ASSOC);

if (!empty($usuarioStatus['bloqueado']) && $usuarioStatus['bloqueado'] == 1) {
    session_destroy();
    session_start();
    $_SESSION['erro_bloqueio'] = '🔒 Seu acesso foi bloqueado. Entre em contato com o suporte.';
    header('Location: ../index.php');
    exit;
}

// Buscar configurações de preço
$configs = [];
$stmt = $db->query("SELECT chave, valor FROM configuracoes");
while ($row = $stmt->fetch(PDO::FETCH_ASSOC)) {
    $configs[$row['chave']] = $row['valor'];
}
$valorMinimo = floatval($configs['valor_minimo_entrega'] ?? 10);

// Preços por categoria
$categorias = [
    'moto' => ['nome' => 'Moto', 'icon' => '🏍️', 'desc' => 'Entregas rápidas', 'base' => floatval($configs['preco_base_moto'] ?? 8), 'km' => floatval($configs['preco_km_moto'] ?? 2.00)],
    'carro' => ['nome' => 'Carro', 'icon' => '🚗', 'desc' => 'Pacotes médios', 'base' => floatval($configs['preco_base_carro'] ?? 12), 'km' => floatval($configs['preco_km_carro'] ?? 2.50)],
    'van' => ['nome' => 'Van', 'icon' => '🚐', 'desc' => 'Volumes maiores', 'base' => floatval($configs['preco_base_van'] ?? 20), 'km' => floatval($configs['preco_km_van'] ?? 3.50)],
    'caminhao' => ['nome' => 'Caminhão', 'icon' => '🚛', 'desc' => 'Cargas pesadas', 'base' => floatval($configs['preco_base_caminhao'] ?? 35), 'km' => floatval($configs['preco_km_caminhao'] ?? 5.00)]
];

// Buscar entregas do usuário
$stmt = $db->prepare("SELECT e.*, ent.nome as entregador_nome, ent.telefone as entregador_telefone FROM entregas e LEFT JOIN entregadores ent ON e.entregador_id = ent.id WHERE e.usuario_id = ? ORDER BY e.created_at DESC LIMIT 10");
$stmt->execute([$userId]);
$entregas = $stmt->fetchAll(PDO::FETCH_ASSOC);

// Buscar banners ativos
$banners = [];
try {
    $stmt = $db->query("SELECT * FROM banners WHERE ativo = 1 ORDER BY ordem ASC LIMIT 4");
    $banners = $stmt->fetchAll(PDO::FETCH_ASSOC);
} catch (Exception $e) {}

$googleMapsKey = 'AIzaSyBQyClNByvKjHubTBizQ1GquLjhwSyHz2k';
$erro = '';
$sucesso = '';

// Processar nova entrega
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $origem = trim($_POST['origem'] ?? '');
    $destino = trim($_POST['destino'] ?? '');
    $descricao = trim($_POST['descricao'] ?? '');
    $origemLat = floatval($_POST['origem_lat'] ?? 0);
    $origemLng = floatval($_POST['origem_lng'] ?? 0);
    $destinoLat = floatval($_POST['destino_lat'] ?? 0);
    $destinoLng = floatval($_POST['destino_lng'] ?? 0);
    $distancia = floatval($_POST['distancia'] ?? 0);
    $categoriaSelecionada = $_POST['categoria'] ?? 'moto';
    
    if (empty($origem) || empty($destino)) {
        $erro = 'Preencha os endereços de origem e destino';
    } elseif (!isset($categorias[$categoriaSelecionada])) {
        $erro = 'Selecione uma categoria válida';
    } else {
        $cat = $categorias[$categoriaSelecionada];
        $valor = $distancia > 0 ? $cat['base'] + ($cat['km'] * $distancia) : $cat['base'] + ($cat['km'] * rand(2, 10));
        $valor = max($valor, $valorMinimo);
        $valor = round($valor, 2);
        
        $stmt = $db->prepare("INSERT INTO entregas (usuario_id, origem_endereco, origem_lat, origem_lng, destino_endereco, destino_lat, destino_lng, descricao, valor, status) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, 'pendente')");
        
        if ($stmt->execute([getCurrentUserId('usuario'), $origem, $origemLat ?: null, $origemLng ?: null, $destino, $destinoLat ?: null, $destinoLng ?: null, $descricao, $valor])) {
            $entregaId = $db->lastInsertId();
            $stmt = $db->prepare("SELECT id FROM entregadores WHERE status = 'disponivel' AND aprovado = 'aprovado' AND (bloqueado = 0 OR bloqueado IS NULL)");
            $stmt->execute();
            $entregadores = $stmt->fetchAll(PDO::FETCH_ASSOC);
            foreach ($entregadores as $ent) {
                $stmt = $db->prepare("INSERT INTO notificacoes (entregador_id, entrega_id, mensagem) VALUES (?, ?, ?)");
                $stmt->execute([$ent['id'], $entregaId, 'Nova entrega disponível! R$ ' . number_format($valor, 2, ',', '.')]);
            }
            $sucesso = 'Entrega solicitada! Valor: R$ ' . number_format($valor, 2, ',', '.');
            // Recarregar entregas
            $stmt = $db->prepare("SELECT e.*, ent.nome as entregador_nome, ent.telefone as entregador_telefone FROM entregas e LEFT JOIN entregadores ent ON e.entregador_id = ent.id WHERE e.usuario_id = ? ORDER BY e.created_at DESC LIMIT 10");
            $stmt->execute([$userId]);
            $entregas = $stmt->fetchAll(PDO::FETCH_ASSOC);
        } else {
            $erro = 'Erro ao solicitar entrega. Tente novamente.';
        }
    }
}
?>
<!DOCTYPE html>
<html lang="pt-BR">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0, maximum-scale=1.0, user-scalable=no">
    <title>Já EntreGo</title>
    <style>
        * { margin: 0; padding: 0; box-sizing: border-box; }
        body { font-family: -apple-system, BlinkMacSystemFont, 'Segoe UI', Roboto, sans-serif; background: #f5f5f5; min-height: 100vh; }
        
        /* Header minimalista */
        .app-header { background: #fff; padding: 12px 16px; display: flex; justify-content: space-between; align-items: center; box-shadow: 0 1px 3px rgba(0,0,0,0.1); position: sticky; top: 0; z-index: 100; }
        .app-header .logo { font-size: 20px; font-weight: 700; color: #1a73e8; }
        .app-header .user-menu { display: flex; align-items: center; gap: 12px; }
        .app-header .user-name { font-size: 14px; color: #666; }
        .app-header .btn-logout { background: none; border: none; color: #999; font-size: 20px; cursor: pointer; }
        
        /* Banner Carrossel */
        .banner-carousel { position: relative; margin: 0; overflow: hidden; border-radius: 0 0 30px 30px; }
        .banner-track { display: flex; transition: transform 0.4s ease; }
        .banner-slide { min-width: 100%; padding: 35px 24px 50px; display: flex; align-items: center; justify-content: space-between; min-height: 200px; }
        .banner-content { flex: 1; padding-right: 20px; }
        .banner-title { font-size: 26px; font-weight: 700; margin-bottom: 10px; line-height: 1.3; }
        .banner-desc { font-size: 16px; opacity: 0.9; line-height: 1.5; }
        .banner-img { width: 140px; height: 140px; object-fit: contain; }
        .banner-dots { position: absolute; bottom: 16px; left: 50%; transform: translateX(-50%); display: flex; gap: 10px; }
        .banner-dot { width: 12px; height: 12px; border-radius: 50%; background: rgba(255,255,255,0.5); cursor: pointer; transition: all 0.3s; }
        .banner-dot.active { background: #fff; transform: scale(1.2); }
        .banner-nav { position: absolute; top: 50%; transform: translateY(-50%); width: 40px; height: 40px; border-radius: 50%; background: rgba(255,255,255,0.95); border: none; cursor: pointer; display: flex; align-items: center; justify-content: center; font-size: 18px; box-shadow: 0 2px 10px rgba(0,0,0,0.2); transition: all 0.3s; }
        .banner-nav:hover { transform: translateY(-50%) scale(1.1); }
        .banner-nav.prev { left: 14px; }
        .banner-nav.next { right: 14px; }
        
        /* Card de Endereços */
        .address-card { background: #fff; margin: 12px; border-radius: 12px; box-shadow: 0 2px 8px rgba(0,0,0,0.08); overflow: hidden; }
        .address-item { display: flex; align-items: flex-start; padding: 14px 16px; border-bottom: 1px solid #f0f0f0; gap: 12px; }
        .address-item:last-child { border-bottom: none; }
        .address-icon { width: 24px; height: 24px; border-radius: 50%; display: flex; align-items: center; justify-content: center; font-size: 12px; flex-shrink: 0; margin-top: 2px; }
        .address-icon.origin { background: #fff3e0; color: #f57c00; }
        .address-icon.dest { background: #ffebee; color: #e53935; }
        .address-content { flex: 1; }
        .address-input { width: 100%; border: none; font-size: 15px; color: #333; background: transparent; outline: none; }
        .address-input::placeholder { color: #999; }
        .address-label { font-size: 11px; color: #999; margin-top: 2px; }
        .address-time { font-size: 13px; color: #1a73e8; font-weight: 500; }
        .btn-location { display: flex; align-items: center; gap: 8px; padding: 12px 16px; color: #1a73e8; font-size: 14px; cursor: pointer; border-top: 1px solid #f0f0f0; background: #fafafa; }
        .btn-location:hover { background: #f0f7ff; }
        .btn-add-stop { display: flex; align-items: center; gap: 8px; padding: 12px 16px; color: #666; font-size: 14px; cursor: pointer; }
        .btn-add-stop:hover { background: #f5f5f5; }
        
        /* Veículos */
        .section-title { padding: 16px 16px 8px; font-size: 15px; font-weight: 600; color: #333; }
        .vehicles-list { padding: 0 12px 12px; }
        .vehicle-item { display: flex; align-items: center; padding: 14px 16px; background: #fff; border-radius: 12px; margin-bottom: 8px; box-shadow: 0 1px 3px rgba(0,0,0,0.06); cursor: pointer; transition: all 0.2s; border: 2px solid transparent; }
        .vehicle-item:hover { box-shadow: 0 2px 8px rgba(0,0,0,0.1); }
        .vehicle-item.selected { border-color: #1a73e8; background: #f0f7ff; }
        .vehicle-icon { width: 48px; height: 48px; background: #fff3e0; border-radius: 10px; display: flex; align-items: center; justify-content: center; font-size: 28px; margin-right: 14px; }
        .vehicle-info { flex: 1; }
        .vehicle-name { font-size: 15px; font-weight: 600; color: #333; }
        .vehicle-desc { font-size: 12px; color: #999; margin-top: 2px; }
        .vehicle-price { text-align: right; }
        .vehicle-price .price { font-size: 16px; font-weight: 700; color: #1a73e8; }
        .vehicle-price .km { font-size: 11px; color: #999; }
        .vehicle-item.disabled { opacity: 0.5; pointer-events: none; }
        .btn-more { text-align: center; padding: 12px; color: #1a73e8; font-size: 14px; cursor: pointer; }
        
        /* Botão Solicitar */
        .btn-submit { display: none; margin: 12px; padding: 16px; background: linear-gradient(135deg, #1a73e8 0%, #0d47a1 100%); color: #fff; border: none; border-radius: 12px; font-size: 16px; font-weight: 600; cursor: pointer; width: calc(100% - 24px); }
        .btn-submit.show { display: block; }
        .btn-submit:hover { transform: translateY(-1px); box-shadow: 0 4px 12px rgba(26,115,232,0.4); }
        
        /* Info Rota */
        .route-info { display: none; margin: 12px; padding: 16px; background: linear-gradient(135deg, #1a73e8 0%, #0d47a1 100%); border-radius: 12px; color: #fff; }
        .route-info.show { display: flex; justify-content: space-around; text-align: center; }
        .route-info .item .value { font-size: 22px; font-weight: 700; }
        .route-info .item .label { font-size: 12px; opacity: 0.8; }
        
        /* Mapa Preview */
        .map-preview { height: 150px; margin: 12px; border-radius: 12px; background: #e0e0e0; display: none; }
        .map-preview.show { display: block; }
        
        /* Minhas Entregas */
        .deliveries-section { margin: 12px; }
        .delivery-item { background: #fff; border-radius: 12px; padding: 14px; margin-bottom: 10px; box-shadow: 0 1px 3px rgba(0,0,0,0.06); }
        .delivery-header { display: flex; justify-content: space-between; align-items: center; margin-bottom: 10px; }
        .delivery-status { padding: 4px 10px; border-radius: 20px; font-size: 11px; font-weight: 600; }
        .delivery-status.pendente { background: #fff3e0; color: #f57c00; }
        .delivery-status.aceita { background: #e3f2fd; color: #1a73e8; }
        .delivery-status.coletada { background: #f3e5f5; color: #9c27b0; }
        .delivery-status.entregue { background: #e8f5e9; color: #4caf50; }
        .delivery-status.cancelada { background: #ffebee; color: #e53935; }
        .delivery-price { font-size: 18px; font-weight: 700; color: #4caf50; }
        .delivery-address { font-size: 13px; color: #666; margin: 4px 0; display: flex; align-items: flex-start; gap: 8px; }
        .delivery-address .icon { flex-shrink: 0; }
        .delivery-driver { background: #f5f5f5; padding: 10px; border-radius: 8px; margin-top: 10px; font-size: 13px; }
        .delivery-time { font-size: 11px; color: #999; margin-top: 8px; }
        
        /* Alertas */
        .alert { margin: 12px; padding: 14px 16px; border-radius: 10px; font-size: 14px; }
        .alert-success { background: #e8f5e9; color: #2e7d32; }
        .alert-danger { background: #ffebee; color: #c62828; }
        
        /* Notificação Popup */
        .notif-popup { position: fixed; top: 70px; right: 16px; left: 16px; background: linear-gradient(135deg, #1a73e8 0%, #0d47a1 100%); color: #fff; padding: 16px; border-radius: 12px; display: none; z-index: 1001; box-shadow: 0 4px 20px rgba(0,0,0,0.3); }
        .notif-popup.show { display: flex; align-items: center; gap: 12px; animation: slideDown 0.3s ease; }
        .notif-popup.entregue { background: linear-gradient(135deg, #4caf50 0%, #2e7d32 100%); }
        .notif-popup .icon { font-size: 32px; }
        .notif-popup .title { font-weight: 600; }
        .notif-popup .text { font-size: 13px; opacity: 0.9; }
        @keyframes slideDown { from { opacity: 0; transform: translateY(-20px); } to { opacity: 1; transform: translateY(0); } }
        
        /* Autocomplete */
        .pac-container { z-index: 10000 !important; border-radius: 8px; box-shadow: 0 4px 20px rgba(0,0,0,0.15); border: none; margin-top: 5px; }
        .pac-item { padding: 10px 15px; cursor: pointer; font-size: 14px; }
        .pac-item:hover { background: #e3f2fd; }
        
        /* Tabs */
        .tabs { display: flex; background: #fff; margin: 12px; border-radius: 10px; overflow: hidden; box-shadow: 0 1px 3px rgba(0,0,0,0.06); }
        .tab { flex: 1; padding: 12px; text-align: center; font-size: 14px; font-weight: 500; color: #666; cursor: pointer; border-bottom: 2px solid transparent; }
        .tab.active { color: #1a73e8; border-bottom-color: #1a73e8; background: #f0f7ff; }
        
        /* Hidden form elements */
        .hidden { display: none; }
    </style>
</head>
<body>

    <!-- Notificação Popup -->
    <div class="notif-popup" id="notificacao">
        <div class="icon" id="notif-icon">🔔</div>
        <div>
            <div class="title" id="notif-titulo">Atualização</div>
            <div class="text" id="notif-texto"></div>
        </div>
    </div>

    <!-- Header -->
    <header class="app-header">
        <div class="logo">🚚 Já EntreGo</div>
        <div class="user-menu">
            <span class="user-name"><?= htmlspecialchars(getCurrentUserName('usuario')) ?></span>
            <a href="logout.php" class="btn-logout" title="Sair">⏻</a>
        </div>
    </header>
    
    <?php if ($erro): ?>
        <div class="alert alert-danger"><?= htmlspecialchars($erro) ?></div>
    <?php endif; ?>
    
    <?php if ($sucesso): ?>
        <div class="alert alert-success"><?= htmlspecialchars($sucesso) ?></div>
    <?php endif; ?>
    
    <!-- Banner Carrossel -->
    <?php if (!empty($banners)): ?>
    <div class="banner-carousel" id="bannerCarousel">
        <button class="banner-nav prev" onclick="moverBanner(-1)">❮</button>
        <div class="banner-track" id="bannerTrack">
            <?php foreach ($banners as $banner): ?>
            <div class="banner-slide" style="background: <?= htmlspecialchars($banner['cor_fundo']) ?>; color: <?= htmlspecialchars($banner['cor_texto']) ?>;">
                <div class="banner-content">
                    <div class="banner-title"><?= htmlspecialchars($banner['titulo']) ?></div>
                    <?php if (!empty($banner['descricao'])): ?>
                    <div class="banner-desc"><?= htmlspecialchars($banner['descricao']) ?></div>
                    <?php endif; ?>
                </div>
                <?php if (!empty($banner['imagem'])): ?>
                <img src="../<?= htmlspecialchars($banner['imagem']) ?>" class="banner-img" alt="">
                <?php endif; ?>
            </div>
            <?php endforeach; ?>
        </div>
        <button class="banner-nav next" onclick="moverBanner(1)">❯</button>
        <div class="banner-dots">
            <?php foreach ($banners as $i => $b): ?>
            <span class="banner-dot <?= $i === 0 ? 'active' : '' ?>" onclick="irParaBanner(<?= $i ?>)"></span>
            <?php endforeach; ?>
        </div>
    </div>
    <?php endif; ?>
    
    <!-- Form de Nova Entrega -->
    <form method="POST" id="formEntrega">
        <!-- Card de Endereços -->
        <div class="address-card">
            <div class="address-item">
                <div class="address-icon origin">●</div>
                <div class="address-content">
                    <input type="text" class="address-input" id="origem" name="origem" placeholder="Endereço de coleta" autocomplete="off" required>
                    <div class="address-label" id="origem-label">Local de coleta</div>
                </div>
                <div class="address-time" id="tempo-coleta">Agora</div>
            </div>
            <div class="address-item">
                <div class="address-icon dest">●</div>
                <div class="address-content">
                    <input type="text" class="address-input" id="destino" name="destino" placeholder="Endereço de entrega" autocomplete="off" required>
                    <div class="address-label">Local de entrega</div>
                </div>
            </div>
            <div class="btn-add-stop" id="btnAddStop">
                <span>＋</span> Adicionar ponto de entrega
            </div>
            <div class="btn-location" onclick="usarMinhaLocalizacao()">
                <svg width="20" height="20" viewBox="0 0 24 24" fill="none" stroke="#1a73e8" stroke-width="2">
                    <circle cx="12" cy="12" r="3"/>
                    <path d="M12 2v4m0 12v4M2 12h4m12 0h4"/>
                </svg>
                Usar minha localização atual
            </div>
        </div>
        
        <input type="hidden" name="origem_lat" id="origem_lat">
        <input type="hidden" name="origem_lng" id="origem_lng">
        <input type="hidden" name="destino_lat" id="destino_lat">
        <input type="hidden" name="destino_lng" id="destino_lng">
        <input type="hidden" name="distancia" id="distancia">
        <input type="hidden" name="categoria" id="categoria" value="moto">
        <input type="hidden" name="descricao" id="descricao" value="">
        
        <!-- Info da Rota -->
        <div class="route-info" id="routeInfo">
            <div class="item">
                <div class="value" id="info-distancia">-</div>
                <div class="label">Distância</div>
            </div>
            <div class="item">
                <div class="value" id="info-tempo">-</div>
                <div class="label">Tempo Est.</div>
            </div>
        </div>
        
        <!-- Mapa Preview -->
        <div class="map-preview" id="mapPreview"></div>
        
        <!-- Veículos Disponíveis -->
        <div class="section-title">Veículos disponíveis</div>
        <div class="vehicles-list" id="vehiclesList">
            <?php foreach ($categorias as $key => $cat): ?>
            <div class="vehicle-item disabled" data-categoria="<?= $key ?>" data-base="<?= $cat['base'] ?>" data-km="<?= $cat['km'] ?>" onclick="selecionarVeiculo('<?= $key ?>')">
                <div class="vehicle-icon"><?= $cat['icon'] ?></div>
                <div class="vehicle-info">
                    <div class="vehicle-name"><?= $cat['nome'] ?></div>
                    <div class="vehicle-desc"><?= $cat['desc'] ?></div>
                </div>
                <div class="vehicle-price">
                    <div class="price" id="preco-<?= $key ?>">--</div>
                    <div class="km">R$ <?= number_format($cat['km'], 2, ',', '.') ?>/km</div>
                </div>
            </div>
            <?php endforeach; ?>
        </div>
        
        <!-- Botão Solicitar -->
        <button type="submit" class="btn-submit" id="btnSubmit">Solicitar Entrega</button>
    </form>

    <!-- Tabs -->
    <div class="tabs">
        <div class="tab active" onclick="mostrarTab('ativas')">Em andamento</div>
        <div class="tab" onclick="mostrarTab('historico')">Histórico</div>
    </div>
    
    <!-- Minhas Entregas -->
    <div class="deliveries-section" id="entregasAtivas">
        <?php 
        $ativas = array_filter($entregas, fn($e) => in_array($e['status'], ['pendente', 'aceita', 'coletada']));
        if (empty($ativas)): ?>
            <div style="text-align: center; padding: 30px; color: #999;">
                <div style="font-size: 40px; margin-bottom: 10px;">📦</div>
                Nenhuma entrega em andamento
            </div>
        <?php else: ?>
            <?php foreach ($ativas as $entrega): ?>
            <div class="delivery-item">
                <div class="delivery-header">
                    <span class="delivery-status <?= $entrega['status'] ?>"><?= ucfirst($entrega['status']) ?></span>
                    <span class="delivery-price">R$ <?= number_format($entrega['valor'], 2, ',', '.') ?></span>
                </div>
                <div class="delivery-address"><span class="icon">📍</span> <?= htmlspecialchars($entrega['origem_endereco']) ?></div>
                <div class="delivery-address"><span class="icon">🎯</span> <?= htmlspecialchars($entrega['destino_endereco']) ?></div>
                <?php if ($entrega['entregador_nome']): ?>
                <div class="delivery-driver">
                    <strong>🏍️ Entregador:</strong> <?= htmlspecialchars($entrega['entregador_nome']) ?>
                    <?php if ($entrega['entregador_telefone']): ?> - <?= htmlspecialchars($entrega['entregador_telefone']) ?><?php endif; ?>
                </div>
                <?php endif; ?>
                <div class="delivery-time"><?= date('d/m/Y H:i', strtotime($entrega['created_at'])) ?></div>
            </div>
            <?php endforeach; ?>
        <?php endif; ?>
    </div>
    
    <div class="deliveries-section hidden" id="entregasHistorico">
        <?php 
        $historico = array_filter($entregas, fn($e) => in_array($e['status'], ['entregue', 'cancelada']));
        if (empty($historico)): ?>
            <div style="text-align: center; padding: 30px; color: #999;">
                <div style="font-size: 40px; margin-bottom: 10px;">📋</div>
                Nenhuma entrega no histórico
            </div>
        <?php else: ?>
            <?php foreach ($historico as $entrega): ?>
            <div class="delivery-item">
                <div class="delivery-header">
                    <span class="delivery-status <?= $entrega['status'] ?>"><?= ucfirst($entrega['status']) ?></span>
                    <span class="delivery-price">R$ <?= number_format($entrega['valor'], 2, ',', '.') ?></span>
                </div>
                <div class="delivery-address"><span class="icon">📍</span> <?= htmlspecialchars($entrega['origem_endereco']) ?></div>
                <div class="delivery-address"><span class="icon">🎯</span> <?= htmlspecialchars($entrega['destino_endereco']) ?></div>
                <div class="delivery-time"><?= date('d/m/Y H:i', strtotime($entrega['created_at'])) ?></div>
            </div>
            <?php endforeach; ?>
        <?php endif; ?>
    </div>
    
    <div style="height: 20px;"></div>
    
    <script src="https://maps.googleapis.com/maps/api/js?key=<?= $googleMapsKey ?>&libraries=places&callback=initApp" async defer></script>
    <script>
    let map, directionsService, directionsRenderer;
    let origemAutocomplete, destinoAutocomplete;
    let origemPlace, destinoPlace;
    const valorMinimo = <?= $valorMinimo ?>;
    const totalBanners = <?= count($banners) ?>;
    let bannerAtual = 0;
    let intervaloCarrossel;
    
    // Tabs
    function mostrarTab(tab) {
        document.querySelectorAll('.tab').forEach((t, i) => {
            t.classList.toggle('active', (tab === 'ativas' && i === 0) || (tab === 'historico' && i === 1));
        });
        document.getElementById('entregasAtivas').classList.toggle('hidden', tab !== 'ativas');
        document.getElementById('entregasHistorico').classList.toggle('hidden', tab !== 'historico');
    }
    
    // Banner Carrossel
    function moverBanner(dir) {
        bannerAtual += dir;
        if (bannerAtual >= totalBanners) bannerAtual = 0;
        if (bannerAtual < 0) bannerAtual = totalBanners - 1;
        atualizarBanner();
        reiniciarAutoPlay();
    }
    function irParaBanner(i) { bannerAtual = i; atualizarBanner(); reiniciarAutoPlay(); }
    function atualizarBanner() {
        const track = document.getElementById('bannerTrack');
        if (track) track.style.transform = `translateX(-${bannerAtual * 100}%)`;
        document.querySelectorAll('.banner-dot').forEach((d, i) => d.classList.toggle('active', i === bannerAtual));
    }
    function autoPlay() { intervaloCarrossel = setInterval(() => { bannerAtual = (bannerAtual + 1) % totalBanners; atualizarBanner(); }, 4000); }
    function reiniciarAutoPlay() { clearInterval(intervaloCarrossel); autoPlay(); }
    if (totalBanners > 1) autoPlay();

    // Selecionar veículo
    function selecionarVeiculo(cat) {
        document.querySelectorAll('.vehicle-item').forEach(v => v.classList.remove('selected'));
        const item = document.querySelector(`[data-categoria="${cat}"]`);
        if (item && !item.classList.contains('disabled')) {
            item.classList.add('selected');
            document.getElementById('categoria').value = cat;
        }
    }
    
    // Atualizar preços dos veículos
    function atualizarPrecos(distanciaKm) {
        document.querySelectorAll('.vehicle-item').forEach(item => {
            const base = parseFloat(item.dataset.base);
            const km = parseFloat(item.dataset.km);
            let valor = base + (km * distanciaKm);
            valor = Math.max(valor, valorMinimo);
            const cat = item.dataset.categoria;
            document.getElementById('preco-' + cat).textContent = 'R$ ' + valor.toFixed(2).replace('.', ',');
            item.classList.remove('disabled');
        });
        // Selecionar moto por padrão
        if (!document.querySelector('.vehicle-item.selected')) {
            selecionarVeiculo('moto');
        }
        document.getElementById('btnSubmit').classList.add('show');
    }
    
    // Inicializar App
    function initApp() {
        directionsService = new google.maps.DirectionsService();
        directionsRenderer = new google.maps.DirectionsRenderer({
            suppressMarkers: false,
            polylineOptions: { strokeColor: '#1a73e8', strokeWeight: 5 }
        });
        
        // Autocomplete origem
        origemAutocomplete = new google.maps.places.Autocomplete(document.getElementById('origem'), {
            types: ['address'],
            componentRestrictions: { country: 'br' }
        });
        origemAutocomplete.addListener('place_changed', function() {
            origemPlace = origemAutocomplete.getPlace();
            if (origemPlace.geometry) {
                document.getElementById('origem_lat').value = origemPlace.geometry.location.lat();
                document.getElementById('origem_lng').value = origemPlace.geometry.location.lng();
                document.getElementById('origem-label').textContent = origemPlace.formatted_address || '';
                calcularRota();
            }
        });
        
        // Autocomplete destino
        destinoAutocomplete = new google.maps.places.Autocomplete(document.getElementById('destino'), {
            types: ['address'],
            componentRestrictions: { country: 'br' }
        });
        destinoAutocomplete.addListener('place_changed', function() {
            destinoPlace = destinoAutocomplete.getPlace();
            if (destinoPlace.geometry) {
                document.getElementById('destino_lat').value = destinoPlace.geometry.location.lat();
                document.getElementById('destino_lng').value = destinoPlace.geometry.location.lng();
                calcularRota();
            }
        });
    }
    
    // Calcular rota
    function calcularRota() {
        const origemLat = document.getElementById('origem_lat').value;
        const origemLng = document.getElementById('origem_lng').value;
        const destinoLat = document.getElementById('destino_lat').value;
        const destinoLng = document.getElementById('destino_lng').value;
        
        if (!origemLat || !destinoLat) return;
        
        const mapaDiv = document.getElementById('mapPreview');
        mapaDiv.classList.add('show');
        
        if (!map) {
            map = new google.maps.Map(mapaDiv, { zoom: 12, disableDefaultUI: true, zoomControl: true });
            directionsRenderer.setMap(map);
        }
        
        directionsService.route({
            origin: { lat: parseFloat(origemLat), lng: parseFloat(origemLng) },
            destination: { lat: parseFloat(destinoLat), lng: parseFloat(destinoLng) },
            travelMode: google.maps.TravelMode.DRIVING
        }, function(result, status) {
            if (status === 'OK') {
                directionsRenderer.setDirections(result);
                const route = result.routes[0].legs[0];
                const distanciaKm = route.distance.value / 1000;
                const tempoMin = Math.ceil(route.duration.value / 60);
                
                document.getElementById('distancia').value = distanciaKm.toFixed(2);
                document.getElementById('routeInfo').classList.add('show');
                document.getElementById('info-distancia').textContent = distanciaKm.toFixed(1) + ' km';
                document.getElementById('info-tempo').textContent = tempoMin + ' min';
                
                atualizarPrecos(distanciaKm);
            }
        });
    }
    
    // Usar minha localização
    function usarMinhaLocalizacao() {
        const btn = document.querySelector('.btn-location');
        const textoOriginal = btn.innerHTML;
        
        if (navigator.geolocation) {
            btn.innerHTML = '🔄 Obtendo localização...';
            
            navigator.geolocation.getCurrentPosition(function(position) {
                const lat = position.coords.latitude;
                const lng = position.coords.longitude;
                
                const geocoder = new google.maps.Geocoder();
                geocoder.geocode({ location: { lat, lng } }, function(results, status) {
                    if (status === 'OK' && results[0]) {
                        const endereco = results[0].formatted_address;
                        document.getElementById('origem').value = endereco;
                        document.getElementById('origem_lat').value = lat;
                        document.getElementById('origem_lng').value = lng;
                        document.getElementById('origem-label').textContent = endereco;
                        
                        origemPlace = {
                            geometry: { location: { lat: () => lat, lng: () => lng } },
                            formatted_address: endereco
                        };
                        
                        btn.innerHTML = '✅ Localização obtida!';
                        setTimeout(() => { btn.innerHTML = textoOriginal; }, 2000);
                        calcularRota();
                    } else {
                        btn.innerHTML = textoOriginal;
                        alert('Não foi possível obter o endereço.');
                    }
                });
            }, function(error) {
                btn.innerHTML = textoOriginal;
                let msg = 'Não foi possível obter sua localização.';
                if (error.code === 1) msg = 'Permissão negada. Habilite nas configurações.';
                alert(msg);
            }, { enableHighAccuracy: true, timeout: 15000 });
        } else {
            alert('Seu navegador não suporta geolocalização.');
        }
    }
    
    // Prevenir Enter nos campos
    document.getElementById('origem').addEventListener('keydown', e => { if (e.key === 'Enter') e.preventDefault(); });
    document.getElementById('destino').addEventListener('keydown', e => { if (e.key === 'Enter') e.preventDefault(); });

    // Sistema de Notificações
    let notificacoesVistas = [];
    let audioContext = null;
    
    function criarAudioContext() {
        if (!audioContext) audioContext = new (window.AudioContext || window.webkitAudioContext)();
        return audioContext;
    }
    
    function tocarSom(tipo) {
        try {
            const ctx = criarAudioContext();
            const freq1 = tipo === 'entregue' ? 523 : 880;
            const freq2 = tipo === 'entregue' ? 659 : 1100;
            
            const osc1 = ctx.createOscillator();
            const gain1 = ctx.createGain();
            osc1.connect(gain1);
            gain1.connect(ctx.destination);
            osc1.frequency.setValueAtTime(freq1, ctx.currentTime);
            osc1.type = 'sine';
            gain1.gain.setValueAtTime(0, ctx.currentTime);
            gain1.gain.linearRampToValueAtTime(0.5, ctx.currentTime + 0.1);
            gain1.gain.linearRampToValueAtTime(0, ctx.currentTime + 0.4);
            osc1.start(ctx.currentTime);
            osc1.stop(ctx.currentTime + 0.4);
            
            setTimeout(() => {
                const osc2 = ctx.createOscillator();
                const gain2 = ctx.createGain();
                osc2.connect(gain2);
                gain2.connect(ctx.destination);
                osc2.frequency.setValueAtTime(freq2, ctx.currentTime);
                osc2.type = 'sine';
                gain2.gain.setValueAtTime(0, ctx.currentTime);
                gain2.gain.linearRampToValueAtTime(0.5, ctx.currentTime + 0.1);
                gain2.gain.linearRampToValueAtTime(0, ctx.currentTime + 0.4);
                osc2.start(ctx.currentTime);
                osc2.stop(ctx.currentTime + 0.4);
            }, 150);
        } catch (e) {}
        
        if ('vibrate' in navigator) navigator.vibrate([200, 100, 200]);
    }
    
    function mostrarNotificacao(mensagem, tipo) {
        const popup = document.getElementById('notificacao');
        const icon = document.getElementById('notif-icon');
        const titulo = document.getElementById('notif-titulo');
        const texto = document.getElementById('notif-texto');
        
        if (tipo === 'aceita') { icon.textContent = '🏍️'; titulo.textContent = 'Entregador a caminho!'; popup.classList.remove('entregue'); }
        else if (tipo === 'coletada') { icon.textContent = '📦'; titulo.textContent = 'Pacote coletado!'; popup.classList.remove('entregue'); }
        else if (tipo === 'entregue') { icon.textContent = '✅'; titulo.textContent = 'Entrega concluída!'; popup.classList.add('entregue'); }
        
        texto.textContent = mensagem;
        popup.classList.add('show');
        setTimeout(() => popup.classList.remove('show'), 6000);
    }
    
    function verificarNotificacoes() {
        fetch('api/verificar-notificacoes.php')
            .then(r => r.json())
            .then(data => {
                if (data.novas && data.novas.length > 0) {
                    data.novas.forEach(notif => {
                        if (!notificacoesVistas.includes(notif.id)) {
                            notificacoesVistas.push(notif.id);
                            tocarSom(notif.tipo);
                            mostrarNotificacao(notif.mensagem, notif.tipo);
                        }
                    });
                    setTimeout(() => location.reload(), 3000);
                }
            })
            .catch(e => {});
    }
    
    if ('Notification' in window && Notification.permission === 'default') Notification.requestPermission();
    verificarNotificacoes();
    setInterval(verificarNotificacoes, 5000);
    </script>
</body>
</html>
