<?php
/**
 * Configuração e funções de email
 * Suporta: SMTP Gmail, função mail() nativa, ou modo de teste (log)
 */

// Configurações de email
define('EMAIL_FROM', 'andre.rj.app@gmail.com');
define('EMAIL_FROM_NAME', 'Já EntreGo');
define('SITE_URL', 'http://localhost'); // Altere para seu domínio

// Configurações SMTP (para Gmail)
define('SMTP_ENABLED', false); // Mude para true para usar SMTP
define('SMTP_HOST', 'smtp.gmail.com');
define('SMTP_PORT', 587);
define('SMTP_USER', 'andre.rj.app@gmail.com'); // Seu email Gmail
define('SMTP_PASS', ''); // Senha de app do Gmail (não a senha normal!)
define('SMTP_SECURE', 'tls');

// Modo de teste - salva emails em arquivo ao invés de enviar
define('EMAIL_TEST_MODE', true); // Mude para false em produção

/**
 * Envia email usando SMTP ou função mail()
 */
function enviarEmail($para, $assunto, $mensagemHtml) {
    // Modo de teste - salva em arquivo
    if (EMAIL_TEST_MODE) {
        return salvarEmailLog($para, $assunto, $mensagemHtml);
    }
    
    // Tentar SMTP primeiro
    if (SMTP_ENABLED && !empty(SMTP_PASS)) {
        return enviarEmailSMTP($para, $assunto, $mensagemHtml);
    }
    
    // Fallback para função mail()
    $headers = "MIME-Version: 1.0\r\n";
    $headers .= "Content-type: text/html; charset=UTF-8\r\n";
    $headers .= "From: " . EMAIL_FROM_NAME . " <" . EMAIL_FROM . ">\r\n";
    $headers .= "Reply-To: " . EMAIL_FROM . "\r\n";
    $headers .= "X-Mailer: PHP/" . phpversion();
    
    return @mail($para, $assunto, $mensagemHtml, $headers);
}

/**
 * Salva email em arquivo de log (modo teste)
 */
function salvarEmailLog($para, $assunto, $mensagemHtml) {
    $logDir = __DIR__ . '/../logs';
    if (!is_dir($logDir)) {
        mkdir($logDir, 0755, true);
    }
    
    $logFile = $logDir . '/emails.log';
    $timestamp = date('Y-m-d H:i:s');
    
    $log = "\n" . str_repeat('=', 60) . "\n";
    $log .= "DATA: $timestamp\n";
    $log .= "PARA: $para\n";
    $log .= "ASSUNTO: $assunto\n";
    $log .= str_repeat('-', 60) . "\n";
    $log .= strip_tags(str_replace(['<br>', '<br/>', '<br />', '</p>', '</li>', '</div>'], "\n", $mensagemHtml)) . "\n";
    $log .= str_repeat('=', 60) . "\n";
    
    file_put_contents($logFile, $log, FILE_APPEND);
    
    // Também salvar versão HTML para visualização
    $htmlFile = $logDir . '/email_' . time() . '_' . md5($para) . '.html';
    file_put_contents($htmlFile, $mensagemHtml);
    
    return true;
}

/**
 * Envia email via SMTP (Gmail)
 * Implementação simples sem bibliotecas externas
 */
function enviarEmailSMTP($para, $assunto, $mensagemHtml) {
    $socket = @fsockopen(SMTP_HOST, SMTP_PORT, $errno, $errstr, 30);
    if (!$socket) {
        error_log("SMTP Error: $errstr ($errno)");
        return false;
    }
    
    // Ler resposta inicial
    fgets($socket, 515);
    
    // EHLO
    fputs($socket, "EHLO " . SMTP_HOST . "\r\n");
    while ($line = fgets($socket, 515)) {
        if (substr($line, 3, 1) == ' ') break;
    }
    
    // STARTTLS
    fputs($socket, "STARTTLS\r\n");
    fgets($socket, 515);
    
    // Habilitar TLS
    stream_socket_enable_crypto($socket, true, STREAM_CRYPTO_METHOD_TLS_CLIENT);
    
    // EHLO novamente após TLS
    fputs($socket, "EHLO " . SMTP_HOST . "\r\n");
    while ($line = fgets($socket, 515)) {
        if (substr($line, 3, 1) == ' ') break;
    }
    
    // AUTH LOGIN
    fputs($socket, "AUTH LOGIN\r\n");
    fgets($socket, 515);
    
    fputs($socket, base64_encode(SMTP_USER) . "\r\n");
    fgets($socket, 515);
    
    fputs($socket, base64_encode(SMTP_PASS) . "\r\n");
    $response = fgets($socket, 515);
    
    if (substr($response, 0, 3) != '235') {
        fclose($socket);
        error_log("SMTP Auth Error: $response");
        return false;
    }
    
    // MAIL FROM
    fputs($socket, "MAIL FROM:<" . EMAIL_FROM . ">\r\n");
    fgets($socket, 515);
    
    // RCPT TO
    fputs($socket, "RCPT TO:<$para>\r\n");
    fgets($socket, 515);
    
    // DATA
    fputs($socket, "DATA\r\n");
    fgets($socket, 515);
    
    // Headers e corpo
    $headers = "From: " . EMAIL_FROM_NAME . " <" . EMAIL_FROM . ">\r\n";
    $headers .= "To: $para\r\n";
    $headers .= "Subject: =?UTF-8?B?" . base64_encode($assunto) . "?=\r\n";
    $headers .= "MIME-Version: 1.0\r\n";
    $headers .= "Content-Type: text/html; charset=UTF-8\r\n";
    $headers .= "\r\n";
    
    fputs($socket, $headers . $mensagemHtml . "\r\n.\r\n");
    fgets($socket, 515);
    
    // QUIT
    fputs($socket, "QUIT\r\n");
    fclose($socket);
    
    return true;
}

/**
 * Template base de email
 */
function templateEmail($titulo, $conteudo) {
    return '
    <!DOCTYPE html>
    <html>
    <head>
        <meta charset="UTF-8">
        <style>
            body { font-family: Arial, sans-serif; background: #f5f5f5; margin: 0; padding: 20px; }
            .container { max-width: 600px; margin: 0 auto; background: white; border-radius: 12px; overflow: hidden; box-shadow: 0 2px 10px rgba(0,0,0,0.1); }
            .header { background: linear-gradient(135deg, #0d47a1 0%, #1a73e8 100%); color: white; padding: 30px; text-align: center; }
            .header h1 { margin: 0; font-size: 28px; }
            .content { padding: 30px; }
            .content h2 { color: #0d47a1; margin-top: 0; }
            .footer { background: #f5f5f5; padding: 20px; text-align: center; color: #666; font-size: 12px; }
            .btn { display: inline-block; background: #0d47a1; color: white; padding: 12px 30px; text-decoration: none; border-radius: 8px; margin: 20px 0; }
            .info-box { background: #e3f2fd; padding: 15px; border-radius: 8px; margin: 15px 0; }
            .warning-box { background: #fff3e0; padding: 15px; border-radius: 8px; margin: 15px 0; border-left: 4px solid #ff9800; }
            .success-box { background: #e8f5e9; padding: 15px; border-radius: 8px; margin: 15px 0; border-left: 4px solid #4caf50; }
        </style>
    </head>
    <body>
        <div class="container">
            <div class="header">
                <h1>🚚 Já EntreGo</h1>
            </div>
            <div class="content">
                <h2>' . $titulo . '</h2>
                ' . $conteudo . '
            </div>
            <div class="footer">
                <p>© ' . date('Y') . ' Já EntreGo - Plataforma de Entregas</p>
                <p>Este é um email automático, não responda.</p>
            </div>
        </div>
    </body>
    </html>';
}

/**
 * Email de confirmação de cadastro (aguardando aprovação)
 */
function enviarEmailCadastroEntregador($email, $nome) {
    $conteudo = '
        <p>Olá <strong>' . htmlspecialchars($nome) . '</strong>,</p>
        <p>Recebemos seu cadastro como entregador na plataforma <strong>Já EntreGo</strong>!</p>
        
        <div class="warning-box">
            <strong>⏳ Aguardando Aprovação</strong><br>
            Seu cadastro está em análise. Nossa equipe irá verificar seus documentos e informações.
            <br><br>
            <strong>Prazo de análise:</strong> até 72 horas úteis.
        </div>
        
        <p>Você receberá um email assim que seu cadastro for aprovado.</p>
        
        <div class="info-box">
            <strong>📋 Próximos passos:</strong>
            <ul>
                <li>Aguarde a análise dos documentos</li>
                <li>Você receberá um email de confirmação</li>
                <li>Após aprovado, faça login e comece a entregar!</li>
            </ul>
        </div>
        
        <p>Obrigado por escolher a Já EntreGo!</p>
    ';
    
    return enviarEmail($email, 'Cadastro Recebido - Já EntreGo', templateEmail('Cadastro Recebido!', $conteudo));
}

/**
 * Email de aprovação do cadastro
 */
function enviarEmailAprovacao($email, $nome) {
    $conteudo = '
        <p>Olá <strong>' . htmlspecialchars($nome) . '</strong>,</p>
        
        <div class="success-box">
            <strong>🎉 Parabéns! Seu cadastro foi APROVADO!</strong>
        </div>
        
        <p>Bem-vindo à família <strong>Já EntreGo</strong>! Agora você faz parte da nossa equipe de entregadores.</p>
        
        <div class="info-box">
            <strong>🚀 Como começar:</strong>
            <ol>
                <li>Acesse o app do entregador</li>
                <li>Faça login com seu email e senha</li>
                <li>Ative seu status para "Online"</li>
                <li>Aguarde as solicitações de entrega</li>
            </ol>
        </div>
        
        <p style="text-align: center;">
            <a href="' . SITE_URL . '/entregador/" class="btn">Acessar App do Entregador</a>
        </p>
        
        <div class="info-box">
            <strong>💡 Dicas para ter sucesso:</strong>
            <ul>
                <li>Mantenha seu status online durante seu horário de trabalho</li>
                <li>Aceite as entregas rapidamente</li>
                <li>Seja cordial com os clientes</li>
                <li>Mantenha seus documentos sempre atualizados</li>
            </ul>
        </div>
        
        <p>Boas entregas! 🏍️</p>
    ';
    
    return enviarEmail($email, '🎉 Cadastro Aprovado - Bem-vindo à Já EntreGo!', templateEmail('Bem-vindo à Já EntreGo!', $conteudo));
}

/**
 * Email de rejeição do cadastro
 */
function enviarEmailRejeicao($email, $nome, $motivo) {
    $conteudo = '
        <p>Olá <strong>' . htmlspecialchars($nome) . '</strong>,</p>
        
        <p>Infelizmente, não foi possível aprovar seu cadastro como entregador na plataforma Já EntreGo.</p>
        
        <div class="warning-box">
            <strong>❌ Motivo da rejeição:</strong><br>
            ' . nl2br(htmlspecialchars($motivo)) . '
        </div>
        
        <p>Você pode corrigir as pendências e realizar um novo cadastro.</p>
        
        <p style="text-align: center;">
            <a href="' . SITE_URL . '/entregador/cadastro.php" class="btn">Fazer Novo Cadastro</a>
        </p>
        
        <p>Se tiver dúvidas, entre em contato conosco.</p>
    ';
    
    return enviarEmail($email, 'Cadastro Não Aprovado - Já EntreGo', templateEmail('Cadastro Não Aprovado', $conteudo));
}


/**
 * Email de boas-vindas para cliente
 */
function enviarEmailBoasVindasCliente($email, $nome) {
    $conteudo = '
        <p>Olá <strong>' . htmlspecialchars($nome) . '</strong>,</p>
        
        <div class="success-box">
            <strong>🎉 Bem-vindo à Já EntreGo!</strong>
        </div>
        
        <p>Seu cadastro foi realizado com sucesso. Agora você pode solicitar entregas de forma rápida e segura!</p>
        
        <div class="info-box">
            <strong>🚀 Como usar:</strong>
            <ol>
                <li>Faça login com seu email e senha</li>
                <li>Clique em "Nova Entrega"</li>
                <li>Informe os endereços de coleta e entrega</li>
                <li>Acompanhe sua entrega em tempo real</li>
            </ol>
        </div>
        
        <p style="text-align: center;">
            <a href="' . SITE_URL . '/" class="btn">Acessar Minha Conta</a>
        </p>
        
        <p>Obrigado por escolher a Já EntreGo!</p>
    ';
    
    return enviarEmail($email, 'Bem-vindo à Já EntreGo! 🚚', templateEmail('Bem-vindo!', $conteudo));
}

/**
 * Email de nova entrega solicitada (para cliente)
 */
function enviarEmailNovaEntrega($email, $nome, $origem, $destino, $valor) {
    $conteudo = '
        <p>Olá <strong>' . htmlspecialchars($nome) . '</strong>,</p>
        
        <p>Sua solicitação de entrega foi registrada com sucesso!</p>
        
        <div class="info-box">
            <strong>📦 Detalhes da Entrega:</strong><br><br>
            <strong>📍 Coleta:</strong> ' . htmlspecialchars($origem) . '<br>
            <strong>🎯 Entrega:</strong> ' . htmlspecialchars($destino) . '<br>
            <strong>💰 Valor:</strong> R$ ' . number_format($valor, 2, ',', '.') . '
        </div>
        
        <p>Você receberá uma notificação assim que um entregador aceitar sua solicitação.</p>
        
        <p style="text-align: center;">
            <a href="' . SITE_URL . '/cliente/dashboard.php" class="btn">Acompanhar Entrega</a>
        </p>
    ';
    
    return enviarEmail($email, 'Entrega Solicitada - Já EntreGo', templateEmail('Entrega Solicitada', $conteudo));
}
