<?php
require_once '../config/database.php';
require_once '../config/auth.php';
requireLogin('entregador', 'index.php');

$db = Database::getInstance()->getConnection();
$entregadorId = getCurrentUserId('entregador');

// Verificar se entregador está bloqueado
$stmt = $db->prepare("SELECT bloqueado, motivo_bloqueio FROM entregadores WHERE id = ?");
$stmt->execute([$entregadorId]);
$entregadorStatus = $stmt->fetch(PDO::FETCH_ASSOC);

if (!empty($entregadorStatus['bloqueado']) && $entregadorStatus['bloqueado'] == 1) {
    // Fazer logout e redirecionar
    session_destroy();
    session_start();
    $_SESSION['erro_bloqueio'] = '🚫 Atenção! Seu acesso à plataforma foi bloqueado temporariamente devido a uma verificação necessária em sua conta. Para entender o motivo e solicitar a liberação, entre em contato com o suporte. Nossa equipe está disponível para orientar você.';
    header('Location: index.php');
    exit;
}

// Buscar status atual
$stmt = $db->prepare("SELECT status, latitude, longitude FROM entregadores WHERE id = ?");
$stmt->execute([$entregadorId]);
$entregador = $stmt->fetch(PDO::FETCH_ASSOC);
$statusAtual = $entregador['status'];

// Buscar entrega em andamento
$stmt = $db->prepare("
    SELECT e.*, u.nome as cliente_nome, u.telefone as cliente_telefone
    FROM entregas e
    JOIN usuarios u ON e.usuario_id = u.id
    WHERE e.entregador_id = ? AND e.status IN ('aceita', 'coletada')
    LIMIT 1
");
$stmt->execute([$entregadorId]);
$entregaAtual = $stmt->fetch(PDO::FETCH_ASSOC);

// Buscar entregas disponíveis (se não tiver entrega em andamento)
$entregasDisponiveis = [];
if (!$entregaAtual && $statusAtual === 'disponivel') {
    $stmt = $db->prepare("
        SELECT e.*, u.nome as cliente_nome
        FROM entregas e
        JOIN usuarios u ON e.usuario_id = u.id
        WHERE e.status = 'pendente'
        ORDER BY e.created_at DESC
    ");
    $stmt->execute();
    $entregasDisponiveis = $stmt->fetchAll(PDO::FETCH_ASSOC);
}

// Ganhos do dia
$stmt = $db->prepare("
    SELECT COALESCE(SUM(valor), 0) as total
    FROM entregas
    WHERE entregador_id = ? AND status = 'entregue' AND DATE(updated_at) = CURDATE()
");
$stmt->execute([$entregadorId]);
$ganhosHoje = $stmt->fetch()['total'];

// API Key do Google Maps
$googleMapsKey = 'AIzaSyBQyClNByvKjHubTBizQ1GquLjhwSyHz2k';
?>
<!DOCTYPE html>
<html lang="pt-BR">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Dashboard - Já EntreGo</title>
    <link rel="stylesheet" href="../assets/css/style.css">
    <link rel="stylesheet" href="../assets/css/entregador.css">
    <style>
        #mapa {
            width: 100%;
            height: 250px;
            border-radius: 12px;
            margin-bottom: 16px;
        }
        .nav-buttons {
            display: flex;
            gap: 10px;
            margin-bottom: 16px;
        }
        .nav-btn {
            flex: 1;
            display: flex;
            align-items: center;
            justify-content: center;
            gap: 8px;
            padding: 14px;
            border: none;
            border-radius: 10px;
            font-size: 14px;
            font-weight: 600;
            cursor: pointer;
            text-decoration: none;
            color: white;
        }
        .nav-btn.waze {
            background: linear-gradient(135deg, #33ccff 0%, #00aaff 100%);
        }
        .nav-btn.gmaps {
            background: linear-gradient(135deg, #4285f4 0%, #34a853 100%);
        }
        .nav-btn img {
            width: 24px;
            height: 24px;
        }
        .destino-atual {
            background: linear-gradient(135deg, #0d47a1 0%, #1565c0 100%);
            color: white;
            padding: 12px 16px;
            border-radius: 10px;
            margin-bottom: 12px;
            display: flex;
            align-items: center;
            gap: 10px;
        }
        .destino-atual .icone {
            font-size: 24px;
        }
        .destino-atual .info {
            flex: 1;
        }
        .destino-atual .label {
            font-size: 11px;
            opacity: 0.8;
            text-transform: uppercase;
        }
        .destino-atual .endereco {
            font-size: 14px;
            font-weight: 500;
        }
        /* Botão de Status */
        .btn-status {
            padding: 10px 20px;
            border: none;
            border-radius: 25px;
            font-size: 14px;
            font-weight: 600;
            cursor: pointer;
            transition: all 0.3s;
        }
        .btn-status.online {
            background: linear-gradient(135deg, #00c853 0%, #00a844 100%);
            color: white;
        }
        .btn-status.offline {
            background: linear-gradient(135deg, #ff5252 0%, #ff1744 100%);
            color: white;
        }
        .btn-status:hover {
            transform: scale(1.05);
            box-shadow: 0 4px 15px rgba(0,0,0,0.2);
        }
    </style>
</head>
<body class="entregador">
    <!-- Header -->
    <header class="header-entregador">
        <h1>
            <span class="status-indicator" style="background: <?= $statusAtual === 'disponivel' ? '#00c853' : '#ff1744' ?>"></span>
            Já EntreGo
        </h1>
        <span><?= htmlspecialchars(getCurrentUserName('entregador')) ?></span>
    </header>
    
    <!-- Notificação Popup -->
    <div class="notificacao-popup" id="notificacao">
        <div class="icon">🔔</div>
        <div class="texto">
            <div class="titulo">Nova Entrega!</div>
            <div class="subtitulo" id="notificacao-texto"></div>
        </div>
    </div>
    
    <main class="main-content">
        <!-- Toggle Status -->
        <div class="status-toggle">
            <span>Status: <strong id="status-texto"><?= $statusAtual === 'disponivel' ? 'Online' : 'Offline' ?></strong></span>
            <button type="button" id="btn-toggle-status" class="btn-status <?= $statusAtual === 'disponivel' ? 'online' : 'offline' ?>">
                <?= $statusAtual === 'disponivel' ? '🟢 Online' : '🔴 Offline' ?>
            </button>
        </div>
        
        <?php if ($entregaAtual): ?>
            <!-- Entrega em andamento com Mapa -->
            <div class="entrega-andamento" style="overflow: visible;">
                <div class="header-status">
                    <h3>Entrega em Andamento</h3>
                    <div class="status"><?= $entregaAtual['status'] === 'aceita' ? '📦 Ir para Coleta' : '🚚 Ir para Entrega' ?></div>
                </div>
                <div class="conteudo">
                    <!-- Destino Atual -->
                    <div class="destino-atual">
                        <div class="icone"><?= $entregaAtual['status'] === 'aceita' ? '📦' : '🏠' ?></div>
                        <div class="info">
                            <div class="label"><?= $entregaAtual['status'] === 'aceita' ? 'Ir para coleta' : 'Ir para entrega' ?></div>
                            <div class="endereco">
                                <?= htmlspecialchars($entregaAtual['status'] === 'aceita' ? $entregaAtual['origem_endereco'] : $entregaAtual['destino_endereco']) ?>
                            </div>
                        </div>
                    </div>
                    
                    <!-- Mapa -->
                    <div id="mapa"></div>
                    
                    <!-- Botões de Navegação -->
                    <div class="nav-buttons">
                        <a href="#" id="btn-waze" class="nav-btn waze" target="_blank">
                            🗺️ Waze
                        </a>
                        <a href="#" id="btn-gmaps" class="nav-btn gmaps" target="_blank">
                            📍 Google Maps
                        </a>
                    </div>
                    
                    <!-- Rota completa -->
                    <div class="rota">
                        <div class="ponto">
                            <div class="ponto-icon origem">A</div>
                            <div class="ponto-info">
                                <div class="ponto-label">Coleta</div>
                                <div class="ponto-endereco"><?= htmlspecialchars($entregaAtual['origem_endereco']) ?></div>
                            </div>
                        </div>
                        <div class="linha-conexao"></div>
                        <div class="ponto">
                            <div class="ponto-icon destino">B</div>
                            <div class="ponto-info">
                                <div class="ponto-label">Entrega</div>
                                <div class="ponto-endereco"><?= htmlspecialchars($entregaAtual['destino_endereco']) ?></div>
                            </div>
                        </div>
                    </div>
                    
                    <div style="background: #e3f2fd; padding: 12px; border-radius: 8px; margin: 16px 0;">
                        <strong>Cliente:</strong> <?= htmlspecialchars($entregaAtual['cliente_nome']) ?>
                        <?php if ($entregaAtual['cliente_telefone']): ?>
                            <br><strong>Tel:</strong> 
                            <a href="tel:<?= htmlspecialchars($entregaAtual['cliente_telefone']) ?>" style="color: #0d47a1;">
                                <?= htmlspecialchars($entregaAtual['cliente_telefone']) ?>
                            </a>
                        <?php endif; ?>
                    </div>
                    
                    <div style="text-align: center; font-size: 24px; font-weight: bold; color: #00c853; margin-bottom: 16px;">
                        R$ <?= number_format($entregaAtual['valor'], 2, ',', '.') ?>
                    </div>
                    
                    <?php if ($entregaAtual['status'] === 'aceita'): ?>
                        <button class="btn-acao-entrega btn-coletar" onclick="atualizarStatus(<?= $entregaAtual['id'] ?>, 'coletada')">
                            ✓ Confirmar Coleta
                        </button>
                    <?php else: ?>
                        <button class="btn-acao-entrega btn-entregar" onclick="atualizarStatus(<?= $entregaAtual['id'] ?>, 'entregue')">
                            ✓ Confirmar Entrega
                        </button>
                    <?php endif; ?>
                </div>
            </div>
            
            <script>
            // Dados da entrega para o mapa
            const entregaAtual = {
                status: '<?= $entregaAtual['status'] ?>',
                origem: '<?= addslashes($entregaAtual['origem_endereco']) ?>',
                destino: '<?= addslashes($entregaAtual['destino_endereco']) ?>',
                origemLat: <?= $entregaAtual['origem_lat'] ?: 'null' ?>,
                origemLng: <?= $entregaAtual['origem_lng'] ?: 'null' ?>,
                destinoLat: <?= $entregaAtual['destino_lat'] ?: 'null' ?>,
                destinoLng: <?= $entregaAtual['destino_lng'] ?: 'null' ?>
            };
            </script>
        <?php else: ?>
            <!-- Ganhos do dia (só mostra quando não tem entrega) -->
            <div class="ganhos-card">
                <div class="label">Ganhos de Hoje</div>
                <div class="valor">R$ <?= number_format($ganhosHoje, 2, ',', '.') ?></div>
                <div class="periodo"><?= date('d/m/Y') ?></div>
            </div>
            
            <?php if ($statusAtual === 'disponivel'): ?>
                <!-- Entregas disponíveis -->
                <div id="entregas-container">
                    <?php if (empty($entregasDisponiveis)): ?>
                        <div class="sem-entregas">
                            <div class="icon">📭</div>
                            <h3>Nenhuma entrega disponível</h3>
                            <p>Aguarde, novas entregas aparecerão aqui</p>
                        </div>
                    <?php else: ?>
                        <?php foreach ($entregasDisponiveis as $entrega): ?>
                            <div class="entrega-card nova" id="entrega-<?= $entrega['id'] ?>">
                                <div class="valor">R$ <?= number_format($entrega['valor'], 2, ',', '.') ?></div>
                                <div class="rota">
                                    <div class="ponto">
                                        <div class="ponto-icon origem">A</div>
                                        <div class="ponto-info">
                                            <div class="ponto-label">Coleta</div>
                                            <div class="ponto-endereco"><?= htmlspecialchars($entrega['origem_endereco']) ?></div>
                                        </div>
                                    </div>
                                    <div class="linha-conexao"></div>
                                    <div class="ponto">
                                        <div class="ponto-icon destino">B</div>
                                        <div class="ponto-info">
                                            <div class="ponto-label">Entrega</div>
                                            <div class="ponto-endereco"><?= htmlspecialchars($entrega['destino_endereco']) ?></div>
                                        </div>
                                    </div>
                                </div>
                                <?php if ($entrega['descricao']): ?>
                                    <div style="color: #666; font-size: 14px; margin-bottom: 16px;">
                                        📦 <?= htmlspecialchars($entrega['descricao']) ?>
                                    </div>
                                <?php endif; ?>
                                <div class="acoes-entrega">
                                    <button class="btn-aceitar" onclick="aceitarEntrega(<?= $entrega['id'] ?>)">Aceitar</button>
                                </div>
                            </div>
                        <?php endforeach; ?>
                    <?php endif; ?>
                </div>
            <?php else: ?>
                <div class="sem-entregas">
                    <div class="icon">😴</div>
                    <h3>Você está offline</h3>
                    <p>Ative o status para receber entregas</p>
                </div>
            <?php endif; ?>
            
            <script>const entregaAtual = null;</script>
        <?php endif; ?>
    </main>
    
    <!-- Bottom Navigation -->
    <nav class="bottom-nav">
        <a href="dashboard.php" class="active">
            <span class="icon">🏠</span>
            Início
        </a>
        <a href="historico.php">
            <span class="icon">📋</span>
            Histórico
        </a>
        <a href="ganhos.php">
            <span class="icon">💰</span>
            Ganhos
        </a>
        <a href="logout.php">
            <span class="icon">🚪</span>
            Sair
        </a>
    </nav>
    
    <!-- Google Maps API -->
    <script src="https://maps.googleapis.com/maps/api/js?key=<?= $googleMapsKey ?>&libraries=places,geometry&callback=initMap" async defer></script>
    
    <script>
    let map, directionsService, directionsRenderer;
    let marcadorEntregador, posicaoAtual;
    
    // Inicializar mapa
    function initMap() {
        if (!entregaAtual) return;
        
        directionsService = new google.maps.DirectionsService();
        directionsRenderer = new google.maps.DirectionsRenderer({
            suppressMarkers: false,
            polylineOptions: {
                strokeColor: '#0d47a1',
                strokeWeight: 5
            }
        });
        
        // Posição padrão (será atualizada com geolocalização)
        const defaultPos = { lat: -23.5505, lng: -46.6333 };
        
        map = new google.maps.Map(document.getElementById('mapa'), {
            zoom: 14,
            center: defaultPos,
            disableDefaultUI: true,
            zoomControl: true,
            styles: [
                { featureType: "poi", stylers: [{ visibility: "off" }] }
            ]
        });
        
        directionsRenderer.setMap(map);
        
        // Obter localização atual
        if (navigator.geolocation) {
            navigator.geolocation.getCurrentPosition(
                (position) => {
                    posicaoAtual = {
                        lat: position.coords.latitude,
                        lng: position.coords.longitude
                    };
                    atualizarRota();
                    atualizarLinksNavegacao();
                    
                    // Atualizar posição no servidor
                    fetch('api/atualizar-localizacao.php', {
                        method: 'POST',
                        headers: {'Content-Type': 'application/x-www-form-urlencoded'},
                        body: `lat=${posicaoAtual.lat}&lng=${posicaoAtual.lng}`
                    });
                },
                () => {
                    // Sem permissão, usar geocoding do endereço
                    geocodeEndereco();
                },
                { enableHighAccuracy: true }
            );
            
            // Atualizar posição continuamente
            navigator.geolocation.watchPosition((position) => {
                posicaoAtual = {
                    lat: position.coords.latitude,
                    lng: position.coords.longitude
                };
                atualizarLinksNavegacao();
            }, null, { enableHighAccuracy: true });
        } else {
            geocodeEndereco();
        }
    }
    
    // Geocodificar endereço se não tiver coordenadas
    function geocodeEndereco() {
        const geocoder = new google.maps.Geocoder();
        const endereco = entregaAtual.status === 'aceita' ? entregaAtual.origem : entregaAtual.destino;
        
        geocoder.geocode({ address: endereco + ', Brasil' }, (results, status) => {
            if (status === 'OK' && results[0]) {
                const loc = results[0].geometry.location;
                map.setCenter(loc);
                
                new google.maps.Marker({
                    position: loc,
                    map: map,
                    title: endereco
                });
            }
        });
    }
    
    // Atualizar rota no mapa
    function atualizarRota() {
        if (!posicaoAtual) return;
        
        const destino = entregaAtual.status === 'aceita' ? entregaAtual.origem : entregaAtual.destino;
        
        directionsService.route({
            origin: posicaoAtual,
            destination: destino + ', Brasil',
            travelMode: google.maps.TravelMode.DRIVING
        }, (result, status) => {
            if (status === 'OK') {
                directionsRenderer.setDirections(result);
            }
        });
    }
    
    // Atualizar links de navegação
    function atualizarLinksNavegacao() {
        const destino = entregaAtual.status === 'aceita' ? entregaAtual.origem : entregaAtual.destino;
        const destinoEncoded = encodeURIComponent(destino);
        
        // Link Waze
        let wazeUrl;
        if (posicaoAtual) {
            wazeUrl = `https://waze.com/ul?q=${destinoEncoded}&navigate=yes`;
        } else {
            wazeUrl = `https://waze.com/ul?q=${destinoEncoded}`;
        }
        document.getElementById('btn-waze').href = wazeUrl;
        
        // Link Google Maps
        let gmapsUrl;
        if (posicaoAtual) {
            gmapsUrl = `https://www.google.com/maps/dir/?api=1&origin=${posicaoAtual.lat},${posicaoAtual.lng}&destination=${destinoEncoded}&travelmode=driving`;
        } else {
            gmapsUrl = `https://www.google.com/maps/search/?api=1&query=${destinoEncoded}`;
        }
        document.getElementById('btn-gmaps').href = gmapsUrl;
    }
    
    // Toggle status online/offline
    let statusOnline = <?= $statusAtual === 'disponivel' ? 'true' : 'false' ?>;
    
    document.getElementById('btn-toggle-status').addEventListener('click', function() {
        const btn = this;
        const novoStatus = statusOnline ? 'offline' : 'disponivel';
        
        btn.disabled = true;
        btn.textContent = '⏳ Aguarde...';
        
        fetch('api/atualizar-status.php', {
            method: 'POST',
            headers: {'Content-Type': 'application/x-www-form-urlencoded'},
            body: 'status=' + novoStatus
        })
        .then(r => r.json())
        .then(data => {
            if (data.success) {
                statusOnline = !statusOnline;
                document.getElementById('status-texto').textContent = statusOnline ? 'Online' : 'Offline';
                btn.textContent = statusOnline ? '🟢 Online' : '🔴 Offline';
                btn.className = 'btn-status ' + (statusOnline ? 'online' : 'offline');
                
                // Mostrar mensagem
                alert(statusOnline ? '✅ Você está online!' : '👋 Você desconectou!');
                
                // Recarregar para mostrar entregas
                setTimeout(() => location.reload(), 300);
            } else {
                alert('Erro ao atualizar status');
                btn.textContent = statusOnline ? '🟢 Online' : '🔴 Offline';
            }
            btn.disabled = false;
        })
        .catch(() => {
            alert('Erro de conexão');
            btn.textContent = statusOnline ? '🟢 Online' : '🔴 Offline';
            btn.disabled = false;
        });
    });
    
    // Aceitar entrega
    function aceitarEntrega(id) {
        fetch('api/aceitar-entrega.php', {
            method: 'POST',
            headers: {'Content-Type': 'application/x-www-form-urlencoded'},
            body: 'entrega_id=' + id
        }).then(r => r.json()).then(data => {
            if (data.success) {
                location.reload();
            } else {
                alert(data.message || 'Erro ao aceitar entrega');
            }
        });
    }
    
    // Atualizar status da entrega
    function atualizarStatus(id, status) {
        fetch('api/atualizar-entrega.php', {
            method: 'POST',
            headers: {'Content-Type': 'application/x-www-form-urlencoded'},
            body: 'entrega_id=' + id + '&status=' + status
        }).then(r => r.json()).then(data => {
            if (data.success) {
                location.reload();
            } else {
                alert(data.message || 'Erro ao atualizar');
            }
        });
    }
    
    // Verificar novas entregas periodicamente
    function verificarNovasEntregas() {
        fetch('api/verificar-notificacoes.php')
            .then(r => r.json())
            .then(data => {
                if (data.novas && data.novas.length > 0) {
                    data.novas.forEach(notif => {
                        mostrarNotificacao(notif.mensagem);
                        tocarSom();
                    });
                    setTimeout(() => location.reload(), 2000);
                }
            });
    }
    
    // Mostrar notificação visual
    function mostrarNotificacao(texto) {
        const notif = document.getElementById('notificacao');
        document.getElementById('notificacao-texto').textContent = texto;
        notif.classList.add('show');
        setTimeout(() => notif.classList.remove('show'), 5000);
    }
    
    // Tocar som de notificação
    function tocarSom() {
        if ('vibrate' in navigator) {
            navigator.vibrate([200, 100, 200]);
        }
        // Som usando Web Audio API
        try {
            const ctx = new (window.AudioContext || window.webkitAudioContext)();
            const osc = ctx.createOscillator();
            const gain = ctx.createGain();
            osc.connect(gain);
            gain.connect(ctx.destination);
            osc.frequency.value = 880;
            osc.type = 'sine';
            gain.gain.setValueAtTime(0.5, ctx.currentTime);
            gain.gain.exponentialRampToValueAtTime(0.01, ctx.currentTime + 0.5);
            osc.start(ctx.currentTime);
            osc.stop(ctx.currentTime + 0.5);
        } catch(e) {}
    }
    
    // Verificar a cada 5 segundos
    setInterval(verificarNovasEntregas, 5000);
    
    // Solicitar permissão para notificações
    if ('Notification' in window && Notification.permission === 'default') {
        Notification.requestPermission();
    }
    </script>
</body>
</html>
