<?php
require_once '../config/database.php';
require_once '../config/auth.php';
require_once '../config/helpers.php';
requireLogin('entregador', 'index.php');

$db = Database::getInstance()->getConnection();
$entregadorId = getCurrentUserId('entregador');

// Buscar taxa atual
$taxaPercentual = floatval(getConfig('taxa_entregador_percentual', 15));
$taxaFixa = floatval(getConfig('taxa_entregador_fixa', 0));

// Ganhos de hoje (bruto)
$stmt = $db->prepare("
    SELECT COALESCE(SUM(valor), 0) as total, COUNT(*) as qtd
    FROM entregas
    WHERE entregador_id = ? AND status = 'entregue' AND DATE(updated_at) = CURDATE()
");
$stmt->execute([$entregadorId]);
$hoje = $stmt->fetch(PDO::FETCH_ASSOC);
$hojeLiquido = calcularGanhoEntregador($hoje['total']);

// Ganhos da semana
$stmt = $db->prepare("
    SELECT COALESCE(SUM(valor), 0) as total, COUNT(*) as qtd
    FROM entregas
    WHERE entregador_id = ? AND status = 'entregue' AND YEARWEEK(updated_at) = YEARWEEK(CURDATE())
");
$stmt->execute([$entregadorId]);
$semana = $stmt->fetch(PDO::FETCH_ASSOC);
$semanaLiquido = calcularGanhoEntregador($semana['total']);

// Ganhos do mês
$stmt = $db->prepare("
    SELECT COALESCE(SUM(valor), 0) as total, COUNT(*) as qtd
    FROM entregas
    WHERE entregador_id = ? AND status = 'entregue' AND MONTH(updated_at) = MONTH(CURDATE()) AND YEAR(updated_at) = YEAR(CURDATE())
");
$stmt->execute([$entregadorId]);
$mes = $stmt->fetch(PDO::FETCH_ASSOC);
$mesLiquido = calcularGanhoEntregador($mes['total']);

// Ganhos por dia (últimos 7 dias)
$stmt = $db->prepare("
    SELECT DATE(updated_at) as dia, SUM(valor) as total
    FROM entregas
    WHERE entregador_id = ? AND status = 'entregue' AND updated_at >= DATE_SUB(CURDATE(), INTERVAL 7 DAY)
    GROUP BY DATE(updated_at)
    ORDER BY dia DESC
");
$stmt->execute([$entregadorId]);
$porDia = $stmt->fetchAll(PDO::FETCH_ASSOC);
?>
<!DOCTYPE html>
<html lang="pt-BR">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Ganhos - Entregador</title>
    <link rel="stylesheet" href="../assets/css/style.css">
    <link rel="stylesheet" href="../assets/css/entregador.css">
</head>
<body class="entregador">
    <header class="header-entregador">
        <h1>💰 Meus Ganhos</h1>
    </header>
    
    <main class="main-content">
        <!-- Info Taxa -->
        <div style="background: #fff3e0; margin: 16px; padding: 12px 16px; border-radius: 8px; font-size: 13px; color: #e65100;">
            💡 Taxa da plataforma: <?= $taxaPercentual ?>%<?= $taxaFixa > 0 ? ' + R$ ' . number_format($taxaFixa, 2, ',', '.') : '' ?> por entrega
        </div>
        
        <!-- Ganhos Hoje -->
        <div class="ganhos-card">
            <div class="label">Ganho Líquido Hoje</div>
            <div class="valor">R$ <?= number_format($hojeLiquido, 2, ',', '.') ?></div>
            <div class="periodo"><?= $hoje['qtd'] ?> entregas</div>
            <div style="font-size: 12px; opacity: 0.7; margin-top: 5px;">Bruto: R$ <?= number_format($hoje['total'], 2, ',', '.') ?></div>
        </div>
        
        <!-- Resumo -->
        <div style="display: flex; gap: 16px; margin: 16px;">
            <div style="flex: 1; background: white; border-radius: 12px; padding: 16px; text-align: center;">
                <div style="color: #666; font-size: 12px;">Esta Semana</div>
                <div style="font-size: 20px; font-weight: bold; color: #0d47a1;">R$ <?= number_format($semanaLiquido, 2, ',', '.') ?></div>
                <div style="font-size: 12px; color: #999;"><?= $semana['qtd'] ?> entregas</div>
            </div>
            <div style="flex: 1; background: white; border-radius: 12px; padding: 16px; text-align: center;">
                <div style="color: #666; font-size: 12px;">Este Mês</div>
                <div style="font-size: 20px; font-weight: bold; color: #0d47a1;">R$ <?= number_format($mesLiquido, 2, ',', '.') ?></div>
                <div style="font-size: 12px; color: #999;"><?= $mes['qtd'] ?> entregas</div>
            </div>
        </div>
        
        <!-- Últimos 7 dias -->
        <div style="background: white; margin: 16px; border-radius: 12px; padding: 16px;">
            <h3 style="margin-bottom: 16px; color: #333;">Últimos 7 dias</h3>
            <?php if (empty($porDia)): ?>
                <p style="color: #666; text-align: center;">Nenhum ganho registrado</p>
            <?php else: ?>
                <?php foreach ($porDia as $dia): ?>
                    <div style="display: flex; justify-content: space-between; padding: 12px 0; border-bottom: 1px solid #eee;">
                        <span style="color: #666;"><?= date('d/m', strtotime($dia['dia'])) ?></span>
                        <span style="font-weight: bold; color: #00c853;">R$ <?= number_format($dia['total'], 2, ',', '.') ?></span>
                    </div>
                <?php endforeach; ?>
            <?php endif; ?>
        </div>
    </main>
    
    <nav class="bottom-nav">
        <a href="dashboard.php">
            <span class="icon">🏠</span>
            Início
        </a>
        <a href="historico.php">
            <span class="icon">📋</span>
            Histórico
        </a>
        <a href="ganhos.php" class="active">
            <span class="icon">💰</span>
            Ganhos
        </a>
        <a href="logout.php">
            <span class="icon">🚪</span>
            Sair
        </a>
    </nav>
</body>
</html>
